/*
   fd2110_reg.h - register definitions for SoftLab-NSK FD2110 board

   Copyright (C) 2017 - 2023 Konstantin Oblaukhov <oblaukhov@sl.iae.nsk.su>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef FD2110_REG_H
#define FD2110_REG_H

#include <linux/types.h>
#include <linux/io.h>

#define __read_reg(csr, addr) ioread32(&csr[addr])
#define __write_reg(csr, addr, v) iowrite32(v, &csr[addr])

typedef struct _FD2110_VDMA {
	uint32_t _unused3000 : 31;
	uint32_t enable : 1;
} __attribute__((packed)) FD2110_VDMA;

#define FD2110_VDMA_A (0x00)

static __always_inline FD2110_VDMA FD2110_VDMA_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_VDMA_A);
	return *((FD2110_VDMA *)&r);
}

static __always_inline void FD2110_VDMA_W(uint32_t *csr, FD2110_VDMA v)
{
	__write_reg(csr, FD2110_VDMA_A, *((uint32_t *)&v));
}

typedef struct _FD2110_VDMADescriptor {
	uint32_t address : 32;
} __attribute__((packed)) FD2110_VDMADescriptor;

#define FD2110_VDMADescriptor_A (0x01)

static __always_inline FD2110_VDMADescriptor FD2110_VDMADescriptor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_VDMADescriptor_A);
	return *((FD2110_VDMADescriptor *)&r);
}

static __always_inline void FD2110_VDMADescriptor_W(uint32_t *csr, FD2110_VDMADescriptor v)
{
	__write_reg(csr, FD2110_VDMADescriptor_A, *((uint32_t *)&v));
}

typedef struct _FD2110_VideoConfig {
	uint32_t io0 : 1;
	uint32_t io1 : 1;
	uint32_t _unused3102 : 30;
} __attribute__((packed)) FD2110_VideoConfig;

#define FD2110_VideoConfig_A (0x03)

static __always_inline FD2110_VideoConfig FD2110_VideoConfig_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_VideoConfig_A);
	return *((FD2110_VideoConfig *)&r);
}

static __always_inline void FD2110_VideoConfig_W(uint32_t *csr, FD2110_VideoConfig v)
{
	__write_reg(csr, FD2110_VideoConfig_A, *((uint32_t *)&v));
}

typedef struct _FD2110_IRQEnable {
	uint32_t io0IRQ : 1;
	uint32_t _unused0301 : 3;
	uint32_t io1IRQ : 1;
	uint32_t _unused1105 : 7;
	uint32_t sync0IRQ : 1;
	uint32_t _unused1313 : 1;
	uint32_t sync1IRQ : 1;
	uint32_t _unused1515 : 1;
	uint32_t sync2IRQ : 1;
	uint32_t _unused1717 : 1;
	uint32_t sync3IRQ : 1;
	uint32_t _unused1919 : 1;
	uint32_t eth0RxIRQ : 1;
	uint32_t eth1RxIRQ : 1;
	uint32_t eth0TxIRQ : 1;
	uint32_t eth1TxIRQ : 1;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD2110_IRQEnable;

#define FD2110_IRQEnable_A (0x04)

static __always_inline FD2110_IRQEnable FD2110_IRQEnable_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_IRQEnable_A);
	return *((FD2110_IRQEnable *)&r);
}

static __always_inline void FD2110_IRQEnable_W(uint32_t *csr, FD2110_IRQEnable v)
{
	__write_reg(csr, FD2110_IRQEnable_A, *((uint32_t *)&v));
}

typedef struct _FD2110_IRQFlags {
	uint32_t io0IRQ : 1;
	uint32_t io0Field : 1;
	uint32_t io0Buffer : 2;
	uint32_t io1IRQ : 1;
	uint32_t io1Field : 1;
	uint32_t io1Buffer : 2;
	uint32_t _unused1108 : 4;
	uint32_t sync0IRQ : 1;
	uint32_t sync0Field : 1;
	uint32_t sync1IRQ : 1;
	uint32_t sync1Field : 1;
	uint32_t sync2IRQ : 1;
	uint32_t sync2Field : 1;
	uint32_t sync3IRQ : 1;
	uint32_t sync3Field : 1;
	uint32_t eth0RxIRQ : 1;
	uint32_t eth1RxIRQ : 1;
	uint32_t eth0TxIRQ : 1;
	uint32_t eth1TxIRQ : 1;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD2110_IRQFlags;

#define FD2110_IRQFlags_A (0x05)

static __always_inline FD2110_IRQFlags FD2110_IRQFlags_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_IRQFlags_A);
	return *((FD2110_IRQFlags *)&r);
}

static __always_inline void FD2110_IRQFlags_W(uint32_t *csr, FD2110_IRQFlags v)
{
	__write_reg(csr, FD2110_IRQFlags_A, *((uint32_t *)&v));
}

typedef struct _FD2110_VideoClockCounterL {
	uint32_t counter : 32;
} __attribute__((packed)) FD2110_VideoClockCounterL;

#define FD2110_VideoClockCounterL_A (0x06)

static __always_inline FD2110_VideoClockCounterL FD2110_VideoClockCounterL_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_VideoClockCounterL_A);
	return *((FD2110_VideoClockCounterL *)&r);
}

typedef struct _FD2110_VideoClockCounterH {
	uint32_t counter : 32;
} __attribute__((packed)) FD2110_VideoClockCounterH;

#define FD2110_VideoClockCounterH_A (0x07)

static __always_inline FD2110_VideoClockCounterH FD2110_VideoClockCounterH_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_VideoClockCounterH_A);
	return *((FD2110_VideoClockCounterH *)&r);
}

#define FD2110_TimeCounterNanoseconds_A (0x8)
#define FD2110_TimeCounterSecondsL_A (0x9)
#define FD2110_TimeCounterSecondsH_A (0xA)

#define FD2110_VDMAIRQEnable_A(n) (0x20 + n * 2)
#define FD2110_VDMAIRQFlags_A(n) (0x21 + n * 2)

typedef struct _FD2110_MonitorStatus {
	uint32_t criticalOverheat : 1;
	uint32_t overheat : 1;
	uint32_t coreVoltageError : 1;
	uint32_t auxVoltageError : 1;
	uint32_t hisCriticalOverheat : 1;
	uint32_t hisOverheat : 1;
	uint32_t hisCoreVoltageError : 1;
	uint32_t hisAuxVoltageError : 1;
	uint32_t jtagPoison : 1;
	uint32_t _unused3109 : 23;
} __attribute__((packed)) FD2110_MonitorStatus;

#define FD2110_MonitorStatus_A (0x81)

static __always_inline FD2110_MonitorStatus FD2110_MonitorStatus_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_MonitorStatus_A);
	return *((FD2110_MonitorStatus *)&r);
}

typedef struct _FD2110_TemperatureMonitor {
	uint32_t temperature : 10;
	uint32_t maxTemperature : 10;
	uint32_t minTemperature : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD2110_TemperatureMonitor;

#define FD2110_TemperatureMonitor_A (0x82)

static __always_inline FD2110_TemperatureMonitor FD2110_TemperatureMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_TemperatureMonitor_A);
	return *((FD2110_TemperatureMonitor *)&r);
}

typedef struct _FD2110_VCoreMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD2110_VCoreMonitor;

#define FD2110_VCoreMonitor_A (0x83)

static __always_inline FD2110_VCoreMonitor FD2110_VCoreMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_VCoreMonitor_A);
	return *((FD2110_VCoreMonitor *)&r);
}

typedef struct _FD2110_VAuxMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD2110_VAuxMonitor;

#define FD2110_VAuxMonitor_A (0x84)

static __always_inline FD2110_VAuxMonitor FD2110_VAuxMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_VAuxMonitor_A);
	return *((FD2110_VAuxMonitor *)&r);
}

typedef struct _FD2110_SPIControl {
	uint32_t txBits : 12;
	uint32_t rxBits : 12;
	uint32_t bus : 8;
} __attribute__((packed)) FD2110_SPIControl;

#define FD2110_SPIControl_A (0x85)

static __always_inline FD2110_SPIControl FD2110_SPIControl_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_SPIControl_A);
	return *((FD2110_SPIControl *)&r);
}

static __always_inline void FD2110_SPIControl_W(uint32_t *csr, FD2110_SPIControl v)
{
	__write_reg(csr, FD2110_SPIControl_A, *((uint32_t *)&v));
}

typedef struct _FD2110_SPIData {
	uint32_t data : 32;
} __attribute__((packed)) FD2110_SPIData;

#define FD2110_SPIData_A (0x86)

static __always_inline FD2110_SPIData FD2110_SPIData_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_SPIData_A);
	return *((FD2110_SPIData *)&r);
}

static __always_inline void FD2110_SPIData_W(uint32_t *csr, FD2110_SPIData v)
{
	__write_reg(csr, FD2110_SPIData_A, *((uint32_t *)&v));
}

typedef struct _FD2110_I2C {
	uint32_t data : 8;
	uint32_t subAddress : 8;
	uint32_t rw : 1;
	uint32_t address : 7;
	uint32_t cs : 1;
	uint32_t nack : 1;
	uint32_t bus : 1;
	uint32_t _unused3027 : 4;
	uint32_t reset : 1;
} __attribute__((packed)) FD2110_I2C;

#define FD2110_I2C_A (0x87)

static __always_inline FD2110_I2C FD2110_I2C_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_I2C_A);
	return *((FD2110_I2C *)&r);
}

static __always_inline void FD2110_I2C_W(uint32_t *csr, FD2110_I2C v)
{
	__write_reg(csr, FD2110_I2C_A, *((uint32_t *)&v));
}

typedef struct _FD2110_SoftID {
	uint32_t id : 32;
} __attribute__((packed)) FD2110_SoftID;

#define FD2110_SoftID_A (0x8F)

static __always_inline FD2110_SoftID FD2110_SoftID_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_SoftID_A);
	return *((FD2110_SoftID *)&r);
}

typedef struct _FD2110_AnalogSync {
	uint32_t hsync : 11;
	uint32_t vsync : 20;
	uint32_t interlaced : 1;
} __attribute__((packed)) FD2110_AnalogSync;

#define FD2110_AnalogSync_A (0x0B)

static __always_inline FD2110_AnalogSync FD2110_AnalogSync_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_AnalogSync_A);
	return *((FD2110_AnalogSync *)&r);
}

typedef struct _FD2110_AnalogVSyncTime {
	uint32_t time : 32;
} __attribute__((packed)) FD2110_AnalogVSyncTime;

#define FD2110_AnalogVSyncTime_A (0x0C)

static __always_inline FD2110_AnalogVSyncTime FD2110_AnalogVSyncTime_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_AnalogVSyncTime_A);
	return *((FD2110_AnalogVSyncTime *)&r);
}

typedef struct _FD2110_VideoDMAAddress {
	uint32_t vsize : 2;
	uint32_t _unused0302 : 2;
	uint32_t vaddress : 12;
	uint32_t _unused1716 : 2;
	uint32_t aaddress : 14;
} __attribute__((packed)) FD2110_VideoDMAAddress;

#define FD2110_VideoDMAAddress_A(n) (0x3E + n * 1)

static __always_inline FD2110_VideoDMAAddress FD2110_VideoDMAAddress_R(uint32_t *csr,
								       unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoDMAAddress_A(n));
	return *((FD2110_VideoDMAAddress *)&r);
}

static __always_inline void FD2110_VideoDMAAddress_W(uint32_t *csr, unsigned int n,
						     FD2110_VideoDMAAddress v)
{
	__write_reg(csr, FD2110_VideoDMAAddress_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoInCS {
	uint32_t cd : 1;
	uint32_t modeLocked : 1;
	uint32_t levelB : 1;
	uint32_t mode : 3;
	uint32_t reset : 1;
	uint32_t formatLocked : 1;
	uint32_t progressive : 1;
	uint32_t forceMode : 1;
	uint32_t _unused1010 : 1;
	uint32_t capture : 1;
	uint32_t captureRaw : 1;
	uint32_t vbiTop : 1;
	uint32_t _unused1414 : 1;
	uint32_t vis10b : 1;
	uint32_t _unused1616 : 1;
	uint32_t ycSwap : 1;
	uint32_t asiPeriod : 2;
	uint32_t numStreams : 2;
	uint32_t _unused2522 : 4;
	uint32_t vancCapture : 1;
	uint32_t _unused3027 : 4;
	uint32_t test : 1;
} __attribute__((packed)) FD2110_VideoInCS;

#define FD2110_VideoInCS_A(n) (0x40 + n * 16)

static __always_inline FD2110_VideoInCS FD2110_VideoInCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInCS_A(n));
	return *((FD2110_VideoInCS *)&r);
}

static __always_inline void FD2110_VideoInCS_W(uint32_t *csr, unsigned int n, FD2110_VideoInCS v)
{
	__write_reg(csr, FD2110_VideoInCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoInLine {
	uint32_t totalLines : 12;
	uint32_t _unused1512 : 4;
	uint32_t activeLines : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD2110_VideoInLine;

#define FD2110_VideoInLine_A(n) (0x41 + n * 16)

static __always_inline FD2110_VideoInLine FD2110_VideoInLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInLine_A(n));
	return *((FD2110_VideoInLine *)&r);
}

typedef struct _FD2110_VideoInPixel {
	uint32_t totalPixels : 13;
	uint32_t _unused1513 : 3;
	uint32_t activePixels : 13;
	uint32_t _unused3129 : 3;
} __attribute__((packed)) FD2110_VideoInPixel;

#define FD2110_VideoInPixel_A(n) (0x42 + n * 16)

static __always_inline FD2110_VideoInPixel FD2110_VideoInPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInPixel_A(n));
	return *((FD2110_VideoInPixel *)&r);
}

typedef struct _FD2110_VideoInIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD2110_VideoInIRQRefTime;

#define FD2110_VideoInIRQRefTime_A(n) (0x44 + n * 16)

static __always_inline FD2110_VideoInIRQRefTime FD2110_VideoInIRQRefTime_R(uint32_t *csr,
									   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInIRQRefTime_A(n));
	return *((FD2110_VideoInIRQRefTime *)&r);
}

typedef struct _FD2110_VideoInVBI {
	uint32_t vbiTopOdd : 8;
	uint32_t vbiBottomOdd : 8;
	uint32_t vbiTopEven : 8;
	uint32_t vbiBottomEven : 8;
} __attribute__((packed)) FD2110_VideoInVBI;

#define FD2110_VideoInVBI_A(n) (0x45 + n * 16)

static __always_inline FD2110_VideoInVBI FD2110_VideoInVBI_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInVBI_A(n));
	return *((FD2110_VideoInVBI *)&r);
}

typedef struct _FD2110_VideoInANCCounter {
	uint32_t counter : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD2110_VideoInANCCounter;

#define FD2110_VideoInANCCounter_A(n) (0x46 + n * 16)

static __always_inline FD2110_VideoInANCCounter FD2110_VideoInANCCounter_R(uint32_t *csr,
									   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInANCCounter_A(n));
	return *((FD2110_VideoInANCCounter *)&r);
}

typedef struct _FD2110_VideoInVPID {
	uint32_t vpid : 32;
} __attribute__((packed)) FD2110_VideoInVPID;

#define FD2110_VideoInVPID_A(n) (0x47 + n * 16)

static __always_inline FD2110_VideoInVPID FD2110_VideoInVPID_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInVPID_A(n));
	return *((FD2110_VideoInVPID *)&r);
}

typedef struct _FD2110_VideoInFrameSize {
	uint32_t size : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD2110_VideoInFrameSize;

#define FD2110_VideoInFrameSize_A(n) (0x48 + n * 16)

static __always_inline FD2110_VideoInFrameSize FD2110_VideoInFrameSize_R(uint32_t *csr,
									 unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInFrameSize_A(n));
	return *((FD2110_VideoInFrameSize *)&r);
}

typedef struct _FD2110_VideoInVBISize {
	uint32_t size : 25;
	uint32_t _unused3125 : 7;
} __attribute__((packed)) FD2110_VideoInVBISize;

#define FD2110_VideoInVBISize_A(n) (0x4E + n * 16)

static __always_inline FD2110_VideoInVBISize FD2110_VideoInVBISize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInVBISize_A(n));
	return *((FD2110_VideoInVBISize *)&r);
}

typedef struct _FD2110_VideoInVisibleSize {
	uint32_t size : 25;
	uint32_t _unused3125 : 7;
} __attribute__((packed)) FD2110_VideoInVisibleSize;

#define FD2110_VideoInVisibleSize_A(n) (0x4F + n * 16)

static __always_inline FD2110_VideoInVisibleSize FD2110_VideoInVisibleSize_R(uint32_t *csr,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoInVisibleSize_A(n));
	return *((FD2110_VideoInVisibleSize *)&r);
}

typedef struct _FD2110_VideoOutCS {
	uint32_t reset : 1;
	uint32_t mute : 1;
	uint32_t levelB : 1;
	uint32_t mode : 3;
	uint32_t clockM : 1;
	uint32_t freeRunning : 1;
	uint32_t progressive : 1;
	uint32_t audioCount : 2;
	uint32_t playback : 1;
	uint32_t playbackRaw : 1;
	uint32_t vbiTop : 1;
	uint32_t _unused1414 : 1;
	uint32_t vis10b : 1;
	uint32_t _unused1616 : 1;
	uint32_t dataPresent : 1;
	uint32_t syncSource : 1;
	uint32_t _unused1919 : 1;
	uint32_t atcEnable : 1;
	uint32_t atcType : 2;
	uint32_t dpllEnable : 1;
	uint32_t dpllSelect : 1;
	uint32_t watchdogEn : 1;
	uint32_t _unused2926 : 4;
	uint32_t clone : 1;
	uint32_t test : 1;
} __attribute__((packed)) FD2110_VideoOutCS;

#define FD2110_VideoOutCS_A(n) (0x40 + n * 16)

static __always_inline FD2110_VideoOutCS FD2110_VideoOutCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutCS_A(n));
	return *((FD2110_VideoOutCS *)&r);
}

static __always_inline void FD2110_VideoOutCS_W(uint32_t *csr, unsigned int n, FD2110_VideoOutCS v)
{
	__write_reg(csr, FD2110_VideoOutCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoOutLine {
	uint32_t totalLines : 12;
	uint32_t _unused3112 : 20;
} __attribute__((packed)) FD2110_VideoOutLine;

#define FD2110_VideoOutLine_A(n) (0x41 + n * 16)

static __always_inline FD2110_VideoOutLine FD2110_VideoOutLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutLine_A(n));
	return *((FD2110_VideoOutLine *)&r);
}

static __always_inline void FD2110_VideoOutLine_W(uint32_t *csr, unsigned int n,
						  FD2110_VideoOutLine v)
{
	__write_reg(csr, FD2110_VideoOutLine_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoOutPixel {
	uint32_t totalPixels : 13;
	uint32_t _unused1513 : 3;
	uint32_t activePixels : 13;
	uint32_t _unused3129 : 3;
} __attribute__((packed)) FD2110_VideoOutPixel;

#define FD2110_VideoOutPixel_A(n) (0x42 + n * 16)

static __always_inline FD2110_VideoOutPixel FD2110_VideoOutPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutPixel_A(n));
	return *((FD2110_VideoOutPixel *)&r);
}

static __always_inline void FD2110_VideoOutPixel_W(uint32_t *csr, unsigned int n,
						   FD2110_VideoOutPixel v)
{
	__write_reg(csr, FD2110_VideoOutPixel_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoOutStart {
	uint32_t startOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t startEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD2110_VideoOutStart;

#define FD2110_VideoOutStart_A(n) (0x43 + n * 16)

static __always_inline FD2110_VideoOutStart FD2110_VideoOutStart_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutStart_A(n));
	return *((FD2110_VideoOutStart *)&r);
}

static __always_inline void FD2110_VideoOutStart_W(uint32_t *csr, unsigned int n,
						   FD2110_VideoOutStart v)
{
	__write_reg(csr, FD2110_VideoOutStart_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoOutStop {
	uint32_t stopOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t stopEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD2110_VideoOutStop;

#define FD2110_VideoOutStop_A(n) (0x44 + n * 16)

static __always_inline FD2110_VideoOutStop FD2110_VideoOutStop_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutStop_A(n));
	return *((FD2110_VideoOutStop *)&r);
}

static __always_inline void FD2110_VideoOutStop_W(uint32_t *csr, unsigned int n,
						  FD2110_VideoOutStop v)
{
	__write_reg(csr, FD2110_VideoOutStop_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoOutField {
	uint32_t switchOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t switchEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD2110_VideoOutField;

#define FD2110_VideoOutField_A(n) (0x45 + n * 16)

static __always_inline FD2110_VideoOutField FD2110_VideoOutField_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutField_A(n));
	return *((FD2110_VideoOutField *)&r);
}

static __always_inline void FD2110_VideoOutField_W(uint32_t *csr, unsigned int n,
						   FD2110_VideoOutField v)
{
	__write_reg(csr, FD2110_VideoOutField_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoOutAudio {
	uint32_t address : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD2110_VideoOutAudio;

#define FD2110_VideoOutAudio_A(n) (0x46 + n * 16)

static __always_inline FD2110_VideoOutAudio FD2110_VideoOutAudio_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutAudio_A(n));
	return *((FD2110_VideoOutAudio *)&r);
}

typedef struct _FD2110_VideoOutVPID {
	uint32_t vpid : 32;
} __attribute__((packed)) FD2110_VideoOutVPID;

#define FD2110_VideoOutVPID_A(n) (0x47 + n * 16)

static __always_inline FD2110_VideoOutVPID FD2110_VideoOutVPID_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutVPID_A(n));
	return *((FD2110_VideoOutVPID *)&r);
}

static __always_inline void FD2110_VideoOutVPID_W(uint32_t *csr, unsigned int n,
						  FD2110_VideoOutVPID v)
{
	__write_reg(csr, FD2110_VideoOutVPID_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoOutATC {
	uint32_t data : 30;
	uint32_t address : 2;
} __attribute__((packed)) FD2110_VideoOutATC;

#define FD2110_VideoOutATC_A(n) (0x4B + n * 16)

static __always_inline FD2110_VideoOutATC FD2110_VideoOutATC_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutATC_A(n));
	return *((FD2110_VideoOutATC *)&r);
}

static __always_inline void FD2110_VideoOutATC_W(uint32_t *csr, unsigned int n,
						 FD2110_VideoOutATC v)
{
	__write_reg(csr, FD2110_VideoOutATC_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoOutIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD2110_VideoOutIRQRefTime;

#define FD2110_VideoOutIRQRefTime_A(n) (0x4C + n * 16)

static __always_inline FD2110_VideoOutIRQRefTime FD2110_VideoOutIRQRefTime_R(uint32_t *csr,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutIRQRefTime_A(n));
	return *((FD2110_VideoOutIRQRefTime *)&r);
}

typedef struct _FD2110_VideoOutPhaseShift {
	uint32_t phase : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD2110_VideoOutPhaseShift;

#define FD2110_VideoOutPhaseShift_A(n) (0x4D + n * 16)

static __always_inline FD2110_VideoOutPhaseShift FD2110_VideoOutPhaseShift_R(uint32_t *csr,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VideoOutPhaseShift_A(n));
	return *((FD2110_VideoOutPhaseShift *)&r);
}

static __always_inline void FD2110_VideoOutPhaseShift_W(uint32_t *csr, unsigned int n,
							FD2110_VideoOutPhaseShift v)
{
	__write_reg(csr, FD2110_VideoOutPhaseShift_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_VideoOutWatchdog {
	uint32_t timeout : 16;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD2110_VideoOutWatchdog;

#define FD2110_VideoOutWatchdog_A(n) (0x4E + n * 16)

static __always_inline void FD2110_VideoOutWatchdog_W(uint32_t *csr, unsigned int n,
						      FD2110_VideoOutWatchdog v)
{
	__write_reg(csr, FD2110_VideoOutWatchdog_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_EthernetControl {
	uint32_t reset : 1;
	uint32_t mode : 3;
	uint32_t autoNeg : 1;
	uint32_t locked : 1;
	uint32_t lpm : 1;
	uint32_t sfpTxFault : 1;
	uint32_t sfpTxDisable : 1;
	uint32_t sfpModAbs : 1;
	uint32_t sfpRS0 : 1;
	uint32_t sfpRS1 : 1;
	uint32_t sfpRxLOS : 1;
	uint32_t rxNoMCast : 1;
	uint32_t rxPromisc : 1;
	uint32_t _unused2915 : 15;
	uint32_t txProtection : 1;
	uint32_t loopback : 1;
} __attribute__((packed)) FD2110_EthernetControl;

#define FD2110_EthernetControl_A(n) (0xA0 + n * 16)

static __always_inline FD2110_EthernetControl FD2110_EthernetControl_R(uint32_t *csr,
								       unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_EthernetControl_A(n));
	return *((FD2110_EthernetControl *)&r);
}

static __always_inline void FD2110_EthernetControl_W(uint32_t *csr, unsigned int n,
						     FD2110_EthernetControl v)
{
	__write_reg(csr, FD2110_EthernetControl_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_EthernetStats {
	uint32_t dropped : 16;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD2110_EthernetStats;

#define FD2110_EthernetStats_A(n) (0xA1 + n * 16)

static __always_inline FD2110_EthernetStats FD2110_EthernetStats_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_EthernetStats_A(n));
	return *((FD2110_EthernetStats *)&r);
}

static __always_inline void FD2110_EthernetStats_W(uint32_t *csr, unsigned int n,
						   FD2110_EthernetStats v)
{
	__write_reg(csr, FD2110_EthernetStats_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_EthernetMACL {
	uint32_t mac : 32;
} __attribute__((packed)) FD2110_EthernetMACL;

#define FD2110_EthernetMACL_A(n) (0xA2 + n * 16)

static __always_inline FD2110_EthernetMACL FD2110_EthernetMACL_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_EthernetMACL_A(n));
	return *((FD2110_EthernetMACL *)&r);
}

static __always_inline void FD2110_EthernetMACL_W(uint32_t *csr, unsigned int n,
						  FD2110_EthernetMACL v)
{
	__write_reg(csr, FD2110_EthernetMACL_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_EthernetMACH {
	uint32_t mac : 16;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD2110_EthernetMACH;

#define FD2110_EthernetMACH_A(n) (0xA3 + n * 16)

static __always_inline FD2110_EthernetMACH FD2110_EthernetMACH_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_EthernetMACH_A(n));
	return *((FD2110_EthernetMACH *)&r);
}

static __always_inline void FD2110_EthernetMACH_W(uint32_t *csr, unsigned int n,
						  FD2110_EthernetMACH v)
{
	__write_reg(csr, FD2110_EthernetMACH_A(n), *((uint32_t *)&v));
}
typedef struct _FD2110_SDMAControl {
	uint32_t enableRX0 : 1;
	uint32_t enableRX1 : 1;
	uint32_t enableTX0 : 1;
	uint32_t enableTX1 : 1;
	uint32_t _unused3004 : 27;
	uint32_t enable : 1;
} __attribute__((packed)) FD2110_SDMAControl;

#define FD2110_SDMAControl_A (0xC0)

static __always_inline FD2110_SDMAControl FD2110_SDMAControl_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_SDMAControl_A);
	return *((FD2110_SDMAControl *)&r);
}

static __always_inline void FD2110_SDMAControl_W(uint32_t *csr, FD2110_SDMAControl v)
{
	__write_reg(csr, FD2110_SDMAControl_A, *((uint32_t *)&v));
}

typedef struct _FD2110_SDMAPage {
	uint32_t address : 32;
} __attribute__((packed)) FD2110_SDMAPage;

#define FD2110_SDMAPage_A (0xC1)

static __always_inline FD2110_SDMAPage FD2110_SDMAPage_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_SDMAPage_A);
	return *((FD2110_SDMAPage *)&r);
}

static __always_inline void FD2110_SDMAPage_W(uint32_t *csr, FD2110_SDMAPage v)
{
	__write_reg(csr, FD2110_SDMAPage_A, *((uint32_t *)&v));
}

typedef struct _FD2110_SDMAPointer {
	uint32_t hwPointer : 15;
	uint32_t hwUpdate : 1;
	uint32_t swPointer : 15;
	uint32_t swUpdate : 1;
} __attribute__((packed)) FD2110_SDMAPointer;

#define FD2110_SDMAPointer_A(n) (0xC2 + n * 2)

static __always_inline FD2110_SDMAPointer FD2110_SDMAPointer_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_SDMAPointer_A(n));
	return *((FD2110_SDMAPointer *)&r);
}

static __always_inline void FD2110_SDMAPointer_W(uint32_t *csr, unsigned int n,
						 FD2110_SDMAPointer v)
{
	__write_reg(csr, FD2110_SDMAPointer_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_SDMAStats {
	uint32_t drops : 16;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD2110_SDMAStats;

#define FD2110_SDMAStats_A(n) (0xC3 + n * 2)

static __always_inline FD2110_SDMAStats FD2110_SDMAStats_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_SDMAStats_A(n));
	return *((FD2110_SDMAStats *)&r);
}

static __always_inline void FD2110_SDMAStats_W(uint32_t *csr, unsigned int n, FD2110_SDMAStats v)
{
	__write_reg(csr, FD2110_SDMAStats_A(n), *((uint32_t *)&v));
}

typedef struct _FD2110_StaticVersion {
	uint32_t revision : 16;
	uint32_t major : 8;
	uint32_t hwSubRevision : 2;
	uint32_t hwRevision : 3;
	uint32_t hwBoard : 3;
} __attribute__((packed)) FD2110_StaticVersion;

#define FD2110_StaticVersion_A (0xFF)

static __always_inline FD2110_StaticVersion FD2110_StaticVersion_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD2110_StaticVersion_A);
	return *((FD2110_StaticVersion *)&r);
}

typedef struct _FD2110_VDMARxVideoControl {
	uint32_t enable : 1;
	uint32_t size : 2;
	uint32_t _unused0303 : 1;
	uint32_t address : 12;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD2110_VDMARxVideoControl;

#define FD2110_VDMARxVideoControl_A(x, n) (0x1000 + x * 4096 + 0x00 + n * 8)

static __always_inline FD2110_VDMARxVideoControl FD2110_VDMARxVideoControl_R(uint32_t *csr,
									     unsigned int x,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxVideoControl_A(x, n));
	return *((FD2110_VDMARxVideoControl *)&r);
}

static __always_inline void FD2110_VDMARxVideoControl_W(uint32_t *csr, unsigned int x,
							unsigned int n, FD2110_VDMARxVideoControl v)
{
	__write_reg(csr, FD2110_VDMARxVideoControl_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMARxVideoFormat {
	uint32_t stride : 16;
	uint32_t inColor : 2;
	uint32_t inBPP : 2;
	uint32_t outColor : 2;
	uint32_t outBPP : 2;
	uint32_t interlace : 1;
	uint32_t _unused3125 : 7;
} __attribute__((packed)) FD2110_VDMARxVideoFormat;

#define FD2110_VDMARxVideoFormat_A(x, n) (0x1000 + x * 4096 + 0x01 + n * 8)

static __always_inline FD2110_VDMARxVideoFormat FD2110_VDMARxVideoFormat_R(uint32_t *csr,
									   unsigned int x,
									   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxVideoFormat_A(x, n));
	return *((FD2110_VDMARxVideoFormat *)&r);
}

static __always_inline void FD2110_VDMARxVideoFormat_W(uint32_t *csr, unsigned int x,
						       unsigned int n, FD2110_VDMARxVideoFormat v)
{
	__write_reg(csr, FD2110_VDMARxVideoFormat_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMARxVideoStats {
	uint32_t noseq : 16;
	uint32_t _unused3016 : 15;
	uint32_t overflow : 1;
} __attribute__((packed)) FD2110_VDMARxVideoStats;

#define FD2110_VDMARxVideoStats_A(x, n) (0x1000 + x * 4096 + 0x02 + n * 8)

static __always_inline FD2110_VDMARxVideoStats FD2110_VDMARxVideoStats_R(uint32_t *csr,
									 unsigned int x,
									 unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxVideoStats_A(x, n));
	return *((FD2110_VDMARxVideoStats *)&r);
}

typedef struct _FD2110_VDMARxVideoPosition {
	uint32_t position : 26;
	uint32_t _unused2826 : 3;
	uint32_t field : 1;
	uint32_t buffer : 2;
} __attribute__((packed)) FD2110_VDMARxVideoPosition;

#define FD2110_VDMARxVideoPosition_A(x, n) (0x1000 + x * 4096 + 0x03 + n * 8)

static __always_inline FD2110_VDMARxVideoPosition FD2110_VDMARxVideoPosition_R(uint32_t *csr,
									       unsigned int x,
									       unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxVideoPosition_A(x, n));
	return *((FD2110_VDMARxVideoPosition *)&r);
}

typedef struct _FD2110_VDMARxVideoRTPHeader0 {
	uint32_t header : 32;
} __attribute__((packed)) FD2110_VDMARxVideoRTPHeader0;

#define FD2110_VDMARxVideoRTPHeader0_A(x, n) (0x1000 + x * 4096 + 0x04 + n * 8)

static __always_inline FD2110_VDMARxVideoRTPHeader0 FD2110_VDMARxVideoRTPHeader0_R(uint32_t *csr,
										   unsigned int x,
										   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxVideoRTPHeader0_A(x, n));
	return *((FD2110_VDMARxVideoRTPHeader0 *)&r);
}

typedef struct _FD2110_VDMARxVideoRTPHeader1 {
	uint32_t header : 32;
} __attribute__((packed)) FD2110_VDMARxVideoRTPHeader1;

#define FD2110_VDMARxVideoRTPHeader1_A(x, n) (0x1000 + x * 4096 + 0x05 + n * 8)

static __always_inline FD2110_VDMARxVideoRTPHeader1 FD2110_VDMARxVideoRTPHeader1_R(uint32_t *csr,
										   unsigned int x,
										   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxVideoRTPHeader1_A(x, n));
	return *((FD2110_VDMARxVideoRTPHeader1 *)&r);
}

typedef struct _FD2110_VDMARxVideoRTPHeader2 {
	uint32_t header : 32;
} __attribute__((packed)) FD2110_VDMARxVideoRTPHeader2;

#define FD2110_VDMARxVideoRTPHeader2_A(x, n) (0x1000 + x * 4096 + 0x06 + n * 8)

static __always_inline FD2110_VDMARxVideoRTPHeader2 FD2110_VDMARxVideoRTPHeader2_R(uint32_t *csr,
										   unsigned int x,
										   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxVideoRTPHeader2_A(x, n));
	return *((FD2110_VDMARxVideoRTPHeader2 *)&r);
}

typedef struct _FD2110_VDMARxVideoIRQTime {
	uint32_t time : 32;
} __attribute__((packed)) FD2110_VDMARxVideoIRQTime;

#define FD2110_VDMARxVideoIRQTime_A(x, n) (0x1000 + x * 4096 + 0x07 + n * 8)

static __always_inline FD2110_VDMARxVideoIRQTime FD2110_VDMARxVideoIRQTime_R(uint32_t *csr,
									     unsigned int x,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxVideoIRQTime_A(x, n));
	return *((FD2110_VDMARxVideoIRQTime *)&r);
}

typedef struct _FD2110_VDMARxDataControl {
	uint32_t enable : 1;
	uint32_t size : 2;
	uint32_t _unused0303 : 1;
	uint32_t address : 12;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD2110_VDMARxDataControl;

#define FD2110_VDMARxDataControl_A(x, n) (0x1000 + x * 4096 + 0x80 + n * 8)

static __always_inline FD2110_VDMARxDataControl FD2110_VDMARxDataControl_R(uint32_t *csr,
									   unsigned int x,
									   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxDataControl_A(x, n));
	return *((FD2110_VDMARxDataControl *)&r);
}

static __always_inline void FD2110_VDMARxDataControl_W(uint32_t *csr, unsigned int x,
						       unsigned int n, FD2110_VDMARxDataControl v)
{
	__write_reg(csr, FD2110_VDMARxDataControl_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMARxDataStats {
	uint32_t noseq : 16;
	uint32_t _unused3016 : 15;
	uint32_t overflow : 1;
} __attribute__((packed)) FD2110_VDMARxDataStats;

#define FD2110_VDMARxDataStats_A(x, n) (0x1000 + x * 4096 + 0x82 + n * 8)

static __always_inline FD2110_VDMARxDataStats FD2110_VDMARxDataStats_R(uint32_t *csr,
								       unsigned int x,
								       unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxDataStats_A(x, n));
	return *((FD2110_VDMARxDataStats *)&r);
}

typedef struct _FD2110_VDMARxDataPosition {
	uint32_t position : 26;
	uint32_t _unused3126 : 6;
} __attribute__((packed)) FD2110_VDMARxDataPosition;

#define FD2110_VDMARxDataPosition_A(x, n) (0x1000 + x * 4096 + 0x83 + n * 8)

static __always_inline FD2110_VDMARxDataPosition FD2110_VDMARxDataPosition_R(uint32_t *csr,
									     unsigned int x,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMARxDataPosition_A(x, n));
	return *((FD2110_VDMARxDataPosition *)&r);
}

typedef struct _FD2110_VDMATxVideoControl {
	uint32_t enable : 1;
	uint32_t size : 2;
	uint32_t _unused0303 : 1;
	uint32_t address : 12;
	uint32_t destination : 6;
	uint32_t _unused3122 : 10;
} __attribute__((packed)) FD2110_VDMATxVideoControl;

#define FD2110_VDMATxVideoControl_A(x, n) (0x1400 + x * 4096 + 0x00 + n * 16)

static __always_inline FD2110_VDMATxVideoControl FD2110_VDMATxVideoControl_R(uint32_t *csr,
									     unsigned int x,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxVideoControl_A(x, n));
	return *((FD2110_VDMATxVideoControl *)&r);
}

static __always_inline void FD2110_VDMATxVideoControl_W(uint32_t *csr, unsigned int x,
							unsigned int n, FD2110_VDMATxVideoControl v)
{
	__write_reg(csr, FD2110_VDMATxVideoControl_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxVideoFormat {
	uint32_t stride : 15;
	uint32_t height : 12;
	uint32_t color : 2;
	uint32_t bpp : 2;
	uint32_t interlace : 1;
} __attribute__((packed)) FD2110_VDMATxVideoFormat;

#define FD2110_VDMATxVideoFormat_A(x, n) (0x1400 + x * 4096 + 0x01 + n * 16)

static __always_inline FD2110_VDMATxVideoFormat FD2110_VDMATxVideoFormat_R(uint32_t *csr,
									   unsigned int x,
									   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxVideoFormat_A(x, n));
	return *((FD2110_VDMATxVideoFormat *)&r);
}

static __always_inline void FD2110_VDMATxVideoFormat_W(uint32_t *csr, unsigned int x,
						       unsigned int n, FD2110_VDMATxVideoFormat v)
{
	__write_reg(csr, FD2110_VDMATxVideoFormat_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxPacketization {
	uint32_t psize : 11;
	uint32_t _unused1111 : 1;
	uint32_t color : 2;
	uint32_t bpp : 2;
	uint32_t pt : 8;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD2110_VDMATxPacketization;

#define FD2110_VDMATxPacketization_A(x, n) (0x1400 + x * 4096 + 0x02 + n * 16)

static __always_inline FD2110_VDMATxPacketization FD2110_VDMATxPacketization_R(uint32_t *csr,
									       unsigned int x,
									       unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxPacketization_A(x, n));
	return *((FD2110_VDMATxPacketization *)&r);
}

static __always_inline void FD2110_VDMATxPacketization_W(uint32_t *csr, unsigned int x,
							 unsigned int n,
							 FD2110_VDMATxPacketization v)
{
	__write_reg(csr, FD2110_VDMATxPacketization_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxST2110TimingsPacket {
	uint32_t period : 24;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD2110_VDMATxST2110TimingsPacket;

#define FD2110_VDMATxST2110TimingsPacket_A(x, n) (0x1400 + x * 4096 + 0x03 + n * 16)

static __always_inline FD2110_VDMATxST2110TimingsPacket
FD2110_VDMATxST2110TimingsPacket_R(uint32_t *csr, unsigned int x, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxST2110TimingsPacket_A(x, n));
	return *((FD2110_VDMATxST2110TimingsPacket *)&r);
}

static __always_inline void FD2110_VDMATxST2110TimingsPacket_W(uint32_t *csr, unsigned int x,
							       unsigned int n,
							       FD2110_VDMATxST2110TimingsPacket v)
{
	__write_reg(csr, FD2110_VDMATxST2110TimingsPacket_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxST2110TimingsField {
	uint32_t period : 25;
	uint32_t _unused3125 : 7;
} __attribute__((packed)) FD2110_VDMATxST2110TimingsField;

#define FD2110_VDMATxST2110TimingsField_A(x, n) (0x1400 + x * 4096 + 0x04 + n * 16)

static __always_inline FD2110_VDMATxST2110TimingsField
FD2110_VDMATxST2110TimingsField_R(uint32_t *csr, unsigned int x, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxST2110TimingsField_A(x, n));
	return *((FD2110_VDMATxST2110TimingsField *)&r);
}

static __always_inline void FD2110_VDMATxST2110TimingsField_W(uint32_t *csr, unsigned int x,
							      unsigned int n,
							      FD2110_VDMATxST2110TimingsField v)
{
	__write_reg(csr, FD2110_VDMATxST2110TimingsField_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxST2110Timings90Field {
	uint32_t period : 15;
	uint32_t _unused3115 : 17;
} __attribute__((packed)) FD2110_VDMATxST2110Timings90Field;

#define FD2110_VDMATxST2110Timings90Field_A(x, n) (0x1400 + x * 4096 + 0x05 + n * 16)

static __always_inline FD2110_VDMATxST2110Timings90Field
FD2110_VDMATxST2110Timings90Field_R(uint32_t *csr, unsigned int x, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxST2110Timings90Field_A(x, n));
	return *((FD2110_VDMATxST2110Timings90Field *)&r);
}

static __always_inline void FD2110_VDMATxST2110Timings90Field_W(uint32_t *csr, unsigned int x,
								unsigned int n,
								FD2110_VDMATxST2110Timings90Field v)
{
	__write_reg(csr, FD2110_VDMATxST2110Timings90Field_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxSync {
	uint32_t time : 32;
} __attribute__((packed)) FD2110_VDMATxSync;

#define FD2110_VDMATxSync_A(x, n) (0x1400 + x * 4096 + 0x06 + n * 16)

static __always_inline FD2110_VDMATxSync FD2110_VDMATxSync_R(uint32_t *csr, unsigned int x,
							     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxSync_A(x, n));
	return *((FD2110_VDMATxSync *)&r);
}

static __always_inline void FD2110_VDMATxSync_W(uint32_t *csr, unsigned int x, unsigned int n,
						FD2110_VDMATxSync v)
{
	__write_reg(csr, FD2110_VDMATxSync_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxRTPTimestamp {
	uint32_t time : 32;
} __attribute__((packed)) FD2110_VDMATxRTPTimestamp;

#define FD2110_VDMATxRTPTimestamp_A(x, n) (0x1400 + x * 4096 + 0x07 + n * 16)

static __always_inline FD2110_VDMATxRTPTimestamp FD2110_VDMATxRTPTimestamp_R(uint32_t *csr,
									     unsigned int x,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxRTPTimestamp_A(x, n));
	return *((FD2110_VDMATxRTPTimestamp *)&r);
}

static __always_inline void FD2110_VDMATxRTPTimestamp_W(uint32_t *csr, unsigned int x,
							unsigned int n, FD2110_VDMATxRTPTimestamp v)
{
	__write_reg(csr, FD2110_VDMATxRTPTimestamp_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxRTPSSRC {
	uint32_t ssrc : 32;
} __attribute__((packed)) FD2110_VDMATxRTPSSRC;

#define FD2110_VDMATxRTPSSRC_A(x, n) (0x1400 + x * 4096 + 0x08 + n * 16)

static __always_inline FD2110_VDMATxRTPSSRC FD2110_VDMATxRTPSSRC_R(uint32_t *csr, unsigned int x,
								   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxRTPSSRC_A(x, n));
	return *((FD2110_VDMATxRTPSSRC *)&r);
}

static __always_inline void FD2110_VDMATxRTPSSRC_W(uint32_t *csr, unsigned int x, unsigned int n,
						   FD2110_VDMATxRTPSSRC v)
{
	__write_reg(csr, FD2110_VDMATxRTPSSRC_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxVideoPosition {
	uint32_t position : 26;
	uint32_t _unused2826 : 3;
	uint32_t field : 1;
	uint32_t buffer : 2;
} __attribute__((packed)) FD2110_VDMATxVideoPosition;

#define FD2110_VDMATxVideoPosition_A(x, n) (0x1400 + x * 4096 + 0x09 + n * 16)

static __always_inline FD2110_VDMATxVideoPosition FD2110_VDMATxVideoPosition_R(uint32_t *csr,
									       unsigned int x,
									       unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxVideoPosition_A(x, n));
	return *((FD2110_VDMATxVideoPosition *)&r);
}

typedef struct _FD2110_VDMATxVideoStats {
	uint32_t drop : 16;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD2110_VDMATxVideoStats;

#define FD2110_VDMATxVideoStats_A(x, n) (0x1400 + x * 4096 + 0x0A + n * 16)

static __always_inline FD2110_VDMATxVideoStats FD2110_VDMATxVideoStats_R(uint32_t *csr,
									 unsigned int x,
									 unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxVideoStats_A(x, n));
	return *((FD2110_VDMATxVideoStats *)&r);
}

typedef struct _FD2110_VDMATxSchedulerCS {
	uint32_t enable : 1;
	uint32_t _unused3101 : 31;
} __attribute__((packed)) FD2110_VDMATxSchedulerCS;

#define FD2110_VDMATxSchedulerCS_A(x) (0x1400 + x * 4096 + 0x1FD)

static __always_inline FD2110_VDMATxSchedulerCS FD2110_VDMATxSchedulerCS_R(uint32_t *csr,
									   unsigned int x)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxSchedulerCS_A(x));
	return *((FD2110_VDMATxSchedulerCS *)&r);
}

static __always_inline void FD2110_VDMATxSchedulerCS_W(uint32_t *csr, unsigned int x,
						       FD2110_VDMATxSchedulerCS v)
{
	__write_reg(csr, FD2110_VDMATxSchedulerCS_A(x), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxSchedulerTime {
	uint32_t time : 32;
} __attribute__((packed)) FD2110_VDMATxSchedulerTime;

#define FD2110_VDMATxSchedulerTime_A(x) (0x1400 + x * 4096 + 0x1FE)

static __always_inline FD2110_VDMATxSchedulerTime FD2110_VDMATxSchedulerTime_R(uint32_t *csr,
									       unsigned int x)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxSchedulerTime_A(x));
	return *((FD2110_VDMATxSchedulerTime *)&r);
}

typedef struct _FD2110_VDMATxStats {
	uint32_t drop : 32;
} __attribute__((packed)) FD2110_VDMATxStats;

#define FD2110_VDMATxStats_A(x) (0x1400 + x * 4096 + 0x1FF)

static __always_inline FD2110_VDMATxStats FD2110_VDMATxStats_R(uint32_t *csr, unsigned int x)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxStats_A(x));
	return *((FD2110_VDMATxStats *)&r);
}

typedef struct _FD2110_VDMATxAudioControl {
	uint32_t enable : 1;
	uint32_t size : 2;
	uint32_t _unused0303 : 1;
	uint32_t address : 12;
	uint32_t destination : 6;
	uint32_t _unused3122 : 10;
} __attribute__((packed)) FD2110_VDMATxAudioControl;

#define FD2110_VDMATxAudioControl_A(x, n) (0x1400 + x * 4096 + 0x100 + n * 16)

static __always_inline FD2110_VDMATxAudioControl FD2110_VDMATxAudioControl_R(uint32_t *csr,
									     unsigned int x,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioControl_A(x, n));
	return *((FD2110_VDMATxAudioControl *)&r);
}

static __always_inline void FD2110_VDMATxAudioControl_W(uint32_t *csr, unsigned int x,
							unsigned int n, FD2110_VDMATxAudioControl v)
{
	__write_reg(csr, FD2110_VDMATxAudioControl_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxAudioFormat {
	uint32_t sampleSize : 8;
	uint32_t _unused1508 : 8;
	uint32_t packetSizeDenom : 16;
} __attribute__((packed)) FD2110_VDMATxAudioFormat;

#define FD2110_VDMATxAudioFormat_A(x, n) (0x1400 + x * 4096 + 0x101 + n * 16)

static __always_inline FD2110_VDMATxAudioFormat FD2110_VDMATxAudioFormat_R(uint32_t *csr,
									   unsigned int x,
									   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioFormat_A(x, n));
	return *((FD2110_VDMATxAudioFormat *)&r);
}

static __always_inline void FD2110_VDMATxAudioFormat_W(uint32_t *csr, unsigned int x,
						       unsigned int n, FD2110_VDMATxAudioFormat v)
{
	__write_reg(csr, FD2110_VDMATxAudioFormat_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxAudioPacketization {
	uint32_t _unused1500 : 16;
	uint32_t pt : 8;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD2110_VDMATxAudioPacketization;

#define FD2110_VDMATxAudioPacketization_A(x, n) (0x1400 + x * 4096 + 0x102 + n * 16)

static __always_inline FD2110_VDMATxAudioPacketization
FD2110_VDMATxAudioPacketization_R(uint32_t *csr, unsigned int x, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioPacketization_A(x, n));
	return *((FD2110_VDMATxAudioPacketization *)&r);
}

static __always_inline void FD2110_VDMATxAudioPacketization_W(uint32_t *csr, unsigned int x,
							      unsigned int n,
							      FD2110_VDMATxAudioPacketization v)
{
	__write_reg(csr, FD2110_VDMATxAudioPacketization_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxAudioPacketSize {
	uint32_t packetSizeNum : 16;
	uint32_t packetSizeInt : 10;
	uint32_t _unused3126 : 6;
} __attribute__((packed)) FD2110_VDMATxAudioPacketSize;

#define FD2110_VDMATxAudioPacketSize_A(x, n) (0x1400 + x * 4096 + 0x103 + n * 16)

static __always_inline FD2110_VDMATxAudioPacketSize FD2110_VDMATxAudioPacketSize_R(uint32_t *csr,
										   unsigned int x,
										   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioPacketSize_A(x, n));
	return *((FD2110_VDMATxAudioPacketSize *)&r);
}

static __always_inline void FD2110_VDMATxAudioPacketSize_W(uint32_t *csr, unsigned int x,
							   unsigned int n,
							   FD2110_VDMATxAudioPacketSize v)
{
	__write_reg(csr, FD2110_VDMATxAudioPacketSize_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxAudioTimings {
	uint32_t period : 25;
	uint32_t _unused3125 : 7;
} __attribute__((packed)) FD2110_VDMATxAudioTimings;

#define FD2110_VDMATxAudioTimings_A(x, n) (0x1400 + x * 4096 + 0x104 + n * 16)

static __always_inline FD2110_VDMATxAudioTimings FD2110_VDMATxAudioTimings_R(uint32_t *csr,
									     unsigned int x,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioTimings_A(x, n));
	return *((FD2110_VDMATxAudioTimings *)&r);
}

static __always_inline void FD2110_VDMATxAudioTimings_W(uint32_t *csr, unsigned int x,
							unsigned int n, FD2110_VDMATxAudioTimings v)
{
	__write_reg(csr, FD2110_VDMATxAudioTimings_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxAudioTimingsRTP {
	uint32_t period : 15;
	uint32_t periodIsSample : 1;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD2110_VDMATxAudioTimingsRTP;

#define FD2110_VDMATxAudioTimingsRTP_A(x, n) (0x1400 + x * 4096 + 0x105 + n * 16)

static __always_inline FD2110_VDMATxAudioTimingsRTP FD2110_VDMATxAudioTimingsRTP_R(uint32_t *csr,
										   unsigned int x,
										   unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioTimingsRTP_A(x, n));
	return *((FD2110_VDMATxAudioTimingsRTP *)&r);
}

static __always_inline void FD2110_VDMATxAudioTimingsRTP_W(uint32_t *csr, unsigned int x,
							   unsigned int n,
							   FD2110_VDMATxAudioTimingsRTP v)
{
	__write_reg(csr, FD2110_VDMATxAudioTimingsRTP_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxAudioSync {
	uint32_t time : 32;
} __attribute__((packed)) FD2110_VDMATxAudioSync;

#define FD2110_VDMATxAudioSync_A(x, n) (0x1400 + x * 4096 + 0x106 + n * 16)

static __always_inline FD2110_VDMATxAudioSync FD2110_VDMATxAudioSync_R(uint32_t *csr,
								       unsigned int x,
								       unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioSync_A(x, n));
	return *((FD2110_VDMATxAudioSync *)&r);
}

static __always_inline void FD2110_VDMATxAudioSync_W(uint32_t *csr, unsigned int x, unsigned int n,
						     FD2110_VDMATxAudioSync v)
{
	__write_reg(csr, FD2110_VDMATxAudioSync_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxAudioRTPTimestamp {
	uint32_t time : 32;
} __attribute__((packed)) FD2110_VDMATxAudioRTPTimestamp;

#define FD2110_VDMATxAudioRTPTimestamp_A(x, n) (0x1400 + x * 4096 + 0x107 + n * 16)

static __always_inline FD2110_VDMATxAudioRTPTimestamp
FD2110_VDMATxAudioRTPTimestamp_R(uint32_t *csr, unsigned int x, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioRTPTimestamp_A(x, n));
	return *((FD2110_VDMATxAudioRTPTimestamp *)&r);
}

static __always_inline void FD2110_VDMATxAudioRTPTimestamp_W(uint32_t *csr, unsigned int x,
							     unsigned int n,
							     FD2110_VDMATxAudioRTPTimestamp v)
{
	__write_reg(csr, FD2110_VDMATxAudioRTPTimestamp_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxAudioRTPSSRC {
	uint32_t ssrc : 32;
} __attribute__((packed)) FD2110_VDMATxAudioRTPSSRC;

#define FD2110_VDMATxAudioRTPSSRC_A(x, n) (0x1400 + x * 4096 + 0x108 + n * 16)

static __always_inline FD2110_VDMATxAudioRTPSSRC FD2110_VDMATxAudioRTPSSRC_R(uint32_t *csr,
									     unsigned int x,
									     unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioRTPSSRC_A(x, n));
	return *((FD2110_VDMATxAudioRTPSSRC *)&r);
}

static __always_inline void FD2110_VDMATxAudioRTPSSRC_W(uint32_t *csr, unsigned int x,
							unsigned int n, FD2110_VDMATxAudioRTPSSRC v)
{
	__write_reg(csr, FD2110_VDMATxAudioRTPSSRC_A(x, n), *((uint32_t *)&v));
}

typedef struct _FD2110_VDMATxAudioPosition {
	uint32_t position : 26;
	uint32_t _unused3126 : 6;
} __attribute__((packed)) FD2110_VDMATxAudioPosition;

#define FD2110_VDMATxAudioPosition_A(x, n) (0x1400 + x * 4096 + 0x109 + n * 16)

static __always_inline FD2110_VDMATxAudioPosition FD2110_VDMATxAudioPosition_R(uint32_t *csr,
									       unsigned int x,
									       unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioPosition_A(x, n));
	return *((FD2110_VDMATxAudioPosition *)&r);
}

typedef struct _FD2110_VDMATxAudioStats {
	uint32_t drop : 16;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD2110_VDMATxAudioStats;

#define FD2110_VDMATxAudioStats_A(x, n) (0x1400 + x * 4096 + 0x10A + n * 16)

static __always_inline FD2110_VDMATxAudioStats FD2110_VDMATxAudioStats_R(uint32_t *csr,
									 unsigned int x,
									 unsigned int n)
{
	uint32_t r = __read_reg(csr, FD2110_VDMATxAudioStats_A(x, n));
	return *((FD2110_VDMATxAudioStats *)&r);
}

#endif
