/*
   fd922_reg.h - register definitions driver for SoftLab-NSK FD922 board

   Copyright (C) 2017 - 2024 SoftLab-NSK <forward@softlab.tv>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef FD922_REG_H
#define FD922_REG_H

#include <linux/types.h>
#include <linux/io.h>

#define __read_reg(csr, addr) ioread32(&csr[addr])
#define __write_reg(csr, addr, v) iowrite32(v, &csr[addr])

typedef struct _FD922_VDMA {
	uint32_t _unused3000 : 31;
	uint32_t enable : 1;
} __attribute__((packed)) FD922_VDMA;

#define FD922_VDMA_A (0x00)

__always_inline FD922_VDMA FD922_VDMA_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_VDMA_A);
	return *((FD922_VDMA *)&r);
}

__always_inline void FD922_VDMA_W(uint32_t *csr, FD922_VDMA v)
{
	__write_reg(csr, FD922_VDMA_A, *((uint32_t *)&v));
}

typedef struct _FD922_VDMADescriptor {
	uint32_t address : 32;
} __attribute__((packed)) FD922_VDMADescriptor;

#define FD922_VDMADescriptor_A (0x01)

__always_inline FD922_VDMADescriptor FD922_VDMADescriptor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_VDMADescriptor_A);
	return *((FD922_VDMADescriptor *)&r);
}

__always_inline void FD922_VDMADescriptor_W(uint32_t *csr, FD922_VDMADescriptor v)
{
	__write_reg(csr, FD922_VDMADescriptor_A, *((uint32_t *)&v));
}

typedef struct _FD922_IRQEnable {
	uint32_t in0IRQ : 1;
	uint32_t _unused0201 : 2;
	uint32_t in1IRQ : 1;
	uint32_t _unused0504 : 2;
	uint32_t out0IRQ : 1;
	uint32_t _unused0807 : 2;
	uint32_t out1IRQ : 1;
	uint32_t _unused1110 : 2;
	uint32_t sync0IRQ : 1;
	uint32_t _unused1313 : 1;
	uint32_t sync1IRQ : 1;
	uint32_t _unused3115 : 17;
} __attribute__((packed)) FD922_IRQEnable;

#define FD922_IRQEnable_A (0x04)

__always_inline FD922_IRQEnable FD922_IRQEnable_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_IRQEnable_A);
	return *((FD922_IRQEnable *)&r);
}

__always_inline void FD922_IRQEnable_W(uint32_t *csr, FD922_IRQEnable v)
{
	__write_reg(csr, FD922_IRQEnable_A, *((uint32_t *)&v));
}

typedef struct _FD922_IRQFlags {
	uint32_t in0IRQ : 1;
	uint32_t in0Buffer : 2;
	uint32_t in1IRQ : 1;
	uint32_t in1Buffer : 2;
	uint32_t out0IRQ : 1;
	uint32_t out0Buffer : 1;
	uint32_t _unused0808 : 1;
	uint32_t out1IRQ : 1;
	uint32_t out1Buffer : 1;
	uint32_t _unused1111 : 1;
	uint32_t sync0IRQ : 1;
	uint32_t sync0Field : 1;
	uint32_t sync1IRQ : 1;
	uint32_t sync1Field : 1;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD922_IRQFlags;

#define FD922_IRQFlags_A (0x05)

__always_inline FD922_IRQFlags FD922_IRQFlags_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_IRQFlags_A);
	return *((FD922_IRQFlags *)&r);
}

__always_inline void FD922_IRQFlags_W(uint32_t *csr, FD922_IRQFlags v)
{
	__write_reg(csr, FD922_IRQFlags_A, *((uint32_t *)&v));
}

typedef struct _FD922_AnalogSync {
	uint32_t hsync : 11;
	uint32_t vsync : 20;
	uint32_t _unused3131 : 1;
} __attribute__((packed)) FD922_AnalogSync;

#define FD922_AnalogSync_A (0x06)

__always_inline FD922_AnalogSync FD922_AnalogSync_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_AnalogSync_A);
	return *((FD922_AnalogSync *)&r);
}

__always_inline void FD922_AnalogSync_W(uint32_t *csr, FD922_AnalogSync v)
{
	__write_reg(csr, FD922_AnalogSync_A, *((uint32_t *)&v));
}

typedef struct _FD922_Genlock {
	uint32_t mode : 2;
	uint32_t lock : 1;
	uint32_t m : 1;
	uint32_t _unused0804 : 5;
	uint32_t offset : 23;
} __attribute__((packed)) FD922_Genlock;

#define FD922_Genlock_A (0x07)

__always_inline FD922_Genlock FD922_Genlock_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_Genlock_A);
	return *((FD922_Genlock *)&r);
}

__always_inline void FD922_Genlock_W(uint32_t *csr, FD922_Genlock v)
{
	__write_reg(csr, FD922_Genlock_A, *((uint32_t *)&v));
}

typedef struct _FD922_SyncGenerator {
	uint32_t frameSize : 23;
	uint32_t _unused2923 : 7;
	uint32_t m : 1;
	uint32_t sync : 1;
} __attribute__((packed)) FD922_SyncGenerator;

#define FD922_SyncGenerator_A(n) (0x08 + n * 4)

__always_inline FD922_SyncGenerator FD922_SyncGenerator_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_SyncGenerator_A(n));
	return *((FD922_SyncGenerator *)&r);
}

__always_inline void FD922_SyncGenerator_W(uint32_t *csr, unsigned int n, FD922_SyncGenerator v)
{
	__write_reg(csr, FD922_SyncGenerator_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_SyncGeneratorField {
	uint32_t fieldSize : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD922_SyncGeneratorField;

#define FD922_SyncGeneratorField_A(n) (0x09 + n * 4)

__always_inline FD922_SyncGeneratorField FD922_SyncGeneratorField_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_SyncGeneratorField_A(n));
	return *((FD922_SyncGeneratorField *)&r);
}

__always_inline void FD922_SyncGeneratorField_W(uint32_t *csr, unsigned int n,
						FD922_SyncGeneratorField v)
{
	__write_reg(csr, FD922_SyncGeneratorField_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_SyncGeneratorTime {
	uint32_t time : 32;
} __attribute__((packed)) FD922_SyncGeneratorTime;

#define FD922_SyncGeneratorTime_A(n) (0x0A + n * 4)

__always_inline FD922_SyncGeneratorTime FD922_SyncGeneratorTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_SyncGeneratorTime_A(n));
	return *((FD922_SyncGeneratorTime *)&r);
}

__always_inline void FD922_SyncGeneratorTime_W(uint32_t *csr, unsigned int n,
					       FD922_SyncGeneratorTime v)
{
	__write_reg(csr, FD922_SyncGeneratorTime_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_SyncGeneratorPhase {
	uint32_t phase : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD922_SyncGeneratorPhase;

#define FD922_SyncGeneratorPhase_A(n) (0x0B + n * 4)

__always_inline FD922_SyncGeneratorPhase FD922_SyncGeneratorPhase_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_SyncGeneratorPhase_A(n));
	return *((FD922_SyncGeneratorPhase *)&r);
}

__always_inline void FD922_SyncGeneratorPhase_W(uint32_t *csr, unsigned int n,
						FD922_SyncGeneratorPhase v)
{
	__write_reg(csr, FD922_SyncGeneratorPhase_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_AnalogVSyncTime {
	uint32_t time : 32;
} __attribute__((packed)) FD922_AnalogVSyncTime;

#define FD922_AnalogVSyncTime_A (0x10)

__always_inline FD922_AnalogVSyncTime FD922_AnalogVSyncTime_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_AnalogVSyncTime_A);
	return *((FD922_AnalogVSyncTime *)&r);
}

__always_inline void FD922_AnalogVSyncTime_W(uint32_t *csr, FD922_AnalogVSyncTime v)
{
	__write_reg(csr, FD922_AnalogVSyncTime_A, *((uint32_t *)&v));
}

typedef struct _FD922_PPSLTCControl {
	uint32_t mode : 2;
	uint32_t _unused1502 : 14;
	uint32_t valid : 1;
	uint32_t _unused3117 : 15;
} __attribute__((packed)) FD922_PPSLTCControl;

#define FD922_PPSLTCControl_A (0x11)

__always_inline FD922_PPSLTCControl FD922_PPSLTCControl_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_PPSLTCControl_A);
	return *((FD922_PPSLTCControl *)&r);
}

__always_inline void FD922_PPSLTCControl_W(uint32_t *csr, FD922_PPSLTCControl v)
{
	__write_reg(csr, FD922_PPSLTCControl_A, *((uint32_t *)&v));
}

typedef struct _FD922_PPSLTCTimestamp {
	uint32_t time : 32;
} __attribute__((packed)) FD922_PPSLTCTimestamp;

#define FD922_PPSLTCTimestamp_A (0x12)

__always_inline FD922_PPSLTCTimestamp FD922_PPSLTCTimestamp_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_PPSLTCTimestamp_A);
	return *((FD922_PPSLTCTimestamp *)&r);
}

__always_inline void FD922_PPSLTCTimestamp_W(uint32_t *csr, FD922_PPSLTCTimestamp v)
{
	__write_reg(csr, FD922_PPSLTCTimestamp_A, *((uint32_t *)&v));
}

typedef struct _FD922_PPSLTCDataLow {
	uint32_t data : 32;
} __attribute__((packed)) FD922_PPSLTCDataLow;

#define FD922_PPSLTCDataLow_A (0x13)

__always_inline FD922_PPSLTCDataLow FD922_PPSLTCDataLow_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_PPSLTCDataLow_A);
	return *((FD922_PPSLTCDataLow *)&r);
}

__always_inline void FD922_PPSLTCDataLow_W(uint32_t *csr, FD922_PPSLTCDataLow v)
{
	__write_reg(csr, FD922_PPSLTCDataLow_A, *((uint32_t *)&v));
}

typedef struct _FD922_PPSLTCDataHigh {
	uint32_t data : 32;
} __attribute__((packed)) FD922_PPSLTCDataHigh;

#define FD922_PPSLTCDataHigh_A (0x14)

__always_inline FD922_PPSLTCDataHigh FD922_PPSLTCDataHigh_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_PPSLTCDataHigh_A);
	return *((FD922_PPSLTCDataHigh *)&r);
}

__always_inline void FD922_PPSLTCDataHigh_W(uint32_t *csr, FD922_PPSLTCDataHigh v)
{
	__write_reg(csr, FD922_PPSLTCDataHigh_A, *((uint32_t *)&v));
}

typedef struct _FD922_VideoInCS {
	uint32_t cd : 1;
	uint32_t modeLocked : 1;
	uint32_t levelB : 1;
	uint32_t mode : 3;
	uint32_t reset : 1;
	uint32_t formatLocked : 1;
	uint32_t progressive : 1;
	uint32_t forceMode : 1;
	uint32_t _unused1010 : 1;
	uint32_t capture : 1;
	uint32_t captureRaw : 1;
	uint32_t vbiTop : 1;
	uint32_t vbiBottom : 1;
	uint32_t vis10b : 1;
	uint32_t vbi10b : 1;
	uint32_t ycSwap : 1;
	uint32_t asiPeriod : 2;
	uint32_t _unused2519 : 6;
	uint32_t vancCapture : 1;
	uint32_t _unused3027 : 4;
	uint32_t test : 1;
} __attribute__((packed)) FD922_VideoInCS;

#define FD922_VideoInCS_A(n) (0x40 + n * 16)

__always_inline FD922_VideoInCS FD922_VideoInCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInCS_A(n));
	return *((FD922_VideoInCS *)&r);
}

__always_inline void FD922_VideoInCS_W(uint32_t *csr, unsigned int n, FD922_VideoInCS v)
{
	__write_reg(csr, FD922_VideoInCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInLine {
	uint32_t totalLines : 12;
	uint32_t _unused1512 : 4;
	uint32_t activeLines : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD922_VideoInLine;

#define FD922_VideoInLine_A(n) (0x41 + n * 16)

__always_inline FD922_VideoInLine FD922_VideoInLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInLine_A(n));
	return *((FD922_VideoInLine *)&r);
}

__always_inline void FD922_VideoInLine_W(uint32_t *csr, unsigned int n, FD922_VideoInLine v)
{
	__write_reg(csr, FD922_VideoInLine_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInPixel {
	uint32_t totalPixels : 13;
	uint32_t _unused1513 : 3;
	uint32_t activePixels : 13;
	uint32_t _unused3129 : 3;
} __attribute__((packed)) FD922_VideoInPixel;

#define FD922_VideoInPixel_A(n) (0x42 + n * 16)

__always_inline FD922_VideoInPixel FD922_VideoInPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInPixel_A(n));
	return *((FD922_VideoInPixel *)&r);
}

__always_inline void FD922_VideoInPixel_W(uint32_t *csr, unsigned int n, FD922_VideoInPixel v)
{
	__write_reg(csr, FD922_VideoInPixel_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInIRQTime {
	uint32_t time : 32;
} __attribute__((packed)) FD922_VideoInIRQTime;

#define FD922_VideoInIRQTime_A(n) (0x43 + n * 16)

__always_inline FD922_VideoInIRQTime FD922_VideoInIRQTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInIRQTime_A(n));
	return *((FD922_VideoInIRQTime *)&r);
}

__always_inline void FD922_VideoInIRQTime_W(uint32_t *csr, unsigned int n, FD922_VideoInIRQTime v)
{
	__write_reg(csr, FD922_VideoInIRQTime_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD922_VideoInIRQRefTime;

#define FD922_VideoInIRQRefTime_A(n) (0x44 + n * 16)

__always_inline FD922_VideoInIRQRefTime FD922_VideoInIRQRefTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInIRQRefTime_A(n));
	return *((FD922_VideoInIRQRefTime *)&r);
}

__always_inline void FD922_VideoInIRQRefTime_W(uint32_t *csr, unsigned int n,
					       FD922_VideoInIRQRefTime v)
{
	__write_reg(csr, FD922_VideoInIRQRefTime_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInVBI {
	uint32_t vbiTopOdd : 8;
	uint32_t vbiBottomOdd : 8;
	uint32_t vbiTopEven : 8;
	uint32_t vbiBottomEven : 8;
} __attribute__((packed)) FD922_VideoInVBI;

#define FD922_VideoInVBI_A(n) (0x45 + n * 16)

__always_inline FD922_VideoInVBI FD922_VideoInVBI_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInVBI_A(n));
	return *((FD922_VideoInVBI *)&r);
}

__always_inline void FD922_VideoInVBI_W(uint32_t *csr, unsigned int n, FD922_VideoInVBI v)
{
	__write_reg(csr, FD922_VideoInVBI_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInANCCounter {
	uint32_t counter : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD922_VideoInANCCounter;

#define FD922_VideoInANCCounter_A(n) (0x46 + n * 16)

__always_inline FD922_VideoInANCCounter FD922_VideoInANCCounter_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInANCCounter_A(n));
	return *((FD922_VideoInANCCounter *)&r);
}

__always_inline void FD922_VideoInANCCounter_W(uint32_t *csr, unsigned int n,
					       FD922_VideoInANCCounter v)
{
	__write_reg(csr, FD922_VideoInANCCounter_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInVPID {
	uint32_t vpid : 32;
} __attribute__((packed)) FD922_VideoInVPID;

#define FD922_VideoInVPID_A(n) (0x47 + n * 16)

__always_inline FD922_VideoInVPID FD922_VideoInVPID_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInVPID_A(n));
	return *((FD922_VideoInVPID *)&r);
}

__always_inline void FD922_VideoInVPID_W(uint32_t *csr, unsigned int n, FD922_VideoInVPID v)
{
	__write_reg(csr, FD922_VideoInVPID_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInFrameSize {
	uint32_t size : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD922_VideoInFrameSize;

#define FD922_VideoInFrameSize_A(n) (0x48 + n * 16)

__always_inline FD922_VideoInFrameSize FD922_VideoInFrameSize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInFrameSize_A(n));
	return *((FD922_VideoInFrameSize *)&r);
}

__always_inline void FD922_VideoInFrameSize_W(uint32_t *csr, unsigned int n,
					      FD922_VideoInFrameSize v)
{
	__write_reg(csr, FD922_VideoInFrameSize_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInCorrection {
	uint32_t y : 10;
	uint32_t cb : 10;
	uint32_t cr : 10;
	uint32_t load : 1;
	uint32_t enable : 1;
} __attribute__((packed)) FD922_VideoInCorrection;

#define FD922_VideoInCorrection_A(n) (0x49 + n * 16)

__always_inline FD922_VideoInCorrection FD922_VideoInCorrection_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInCorrection_A(n));
	return *((FD922_VideoInCorrection *)&r);
}

__always_inline void FD922_VideoInCorrection_W(uint32_t *csr, unsigned int n,
					       FD922_VideoInCorrection v)
{
	__write_reg(csr, FD922_VideoInCorrection_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInFIFOStatus {
	uint32_t videoMax : 15;
	uint32_t videoError : 1;
	uint32_t audioMax : 15;
	uint32_t audioError : 1;
} __attribute__((packed)) FD922_VideoInFIFOStatus;

#define FD922_VideoInFIFOStatus_A(n) (0x4A + n * 16)

__always_inline FD922_VideoInFIFOStatus FD922_VideoInFIFOStatus_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInFIFOStatus_A(n));
	return *((FD922_VideoInFIFOStatus *)&r);
}

__always_inline void FD922_VideoInFIFOStatus_W(uint32_t *csr, unsigned int n,
					       FD922_VideoInFIFOStatus v)
{
	__write_reg(csr, FD922_VideoInFIFOStatus_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoInVBISize {
	uint32_t size : 25;
	uint32_t _unused3125 : 7;
} __attribute__((packed)) FD922_VideoInVBISize;

#define FD922_VideoInVBISize_A(n) (0x4E + n * 16)

__always_inline FD922_VideoInVBISize FD922_VideoInVBISize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInVBISize_A(n));
	return *((FD922_VideoInVBISize *)&r);
}

typedef struct _FD922_VideoInVisibleSize {
	uint32_t size : 25;
	uint32_t _unused3125 : 7;
} __attribute__((packed)) FD922_VideoInVisibleSize;

#define FD922_VideoInVisibleSize_A(n) (0x4F + n * 16)

__always_inline FD922_VideoInVisibleSize FD922_VideoInVisibleSize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoInVisibleSize_A(n));
	return *((FD922_VideoInVisibleSize *)&r);
}

typedef struct _FD922_VideoOutCS {
	uint32_t reset : 1;
	uint32_t mute : 1;
	uint32_t levelB : 1;
	uint32_t mode : 3;
	uint32_t clockM : 1;
	uint32_t freeRunning : 1;
	uint32_t progressive : 1;
	uint32_t audioCount : 2;
	uint32_t playback : 1;
	uint32_t playbackRaw : 1;
	uint32_t vbiTop : 1;
	uint32_t vbiBottom : 1;
	uint32_t vis10b : 1;
	uint32_t vbi10b : 1;
	uint32_t dataPresent : 1;
	uint32_t syncSource : 1;
	uint32_t _unused1919 : 1;
	uint32_t atcEnable : 1;
	uint32_t atcType : 2;
	uint32_t dpllEnable : 1;
	uint32_t dpllSelect : 1;
	uint32_t watchdogEn : 1;
	uint32_t _unused2926 : 4;
	uint32_t clone : 1;
	uint32_t test : 1;
} __attribute__((packed)) FD922_VideoOutCS;

#define FD922_VideoOutCS_A(n) (0x60 + n * 16)

__always_inline FD922_VideoOutCS FD922_VideoOutCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutCS_A(n));
	return *((FD922_VideoOutCS *)&r);
}

__always_inline void FD922_VideoOutCS_W(uint32_t *csr, unsigned int n, FD922_VideoOutCS v)
{
	__write_reg(csr, FD922_VideoOutCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutLine {
	uint32_t totalLines : 12;
	uint32_t _unused3112 : 20;
} __attribute__((packed)) FD922_VideoOutLine;

#define FD922_VideoOutLine_A(n) (0x61 + n * 16)

__always_inline FD922_VideoOutLine FD922_VideoOutLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutLine_A(n));
	return *((FD922_VideoOutLine *)&r);
}

__always_inline void FD922_VideoOutLine_W(uint32_t *csr, unsigned int n, FD922_VideoOutLine v)
{
	__write_reg(csr, FD922_VideoOutLine_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutPixel {
	uint32_t totalPixels : 13;
	uint32_t _unused1513 : 3;
	uint32_t activePixels : 13;
	uint32_t _unused3129 : 3;
} __attribute__((packed)) FD922_VideoOutPixel;

#define FD922_VideoOutPixel_A(n) (0x62 + n * 16)

__always_inline FD922_VideoOutPixel FD922_VideoOutPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutPixel_A(n));
	return *((FD922_VideoOutPixel *)&r);
}

__always_inline void FD922_VideoOutPixel_W(uint32_t *csr, unsigned int n, FD922_VideoOutPixel v)
{
	__write_reg(csr, FD922_VideoOutPixel_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutStart {
	uint32_t startOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t startEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD922_VideoOutStart;

#define FD922_VideoOutStart_A(n) (0x63 + n * 16)

__always_inline FD922_VideoOutStart FD922_VideoOutStart_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutStart_A(n));
	return *((FD922_VideoOutStart *)&r);
}

__always_inline void FD922_VideoOutStart_W(uint32_t *csr, unsigned int n, FD922_VideoOutStart v)
{
	__write_reg(csr, FD922_VideoOutStart_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutStop {
	uint32_t stopOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t stopEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD922_VideoOutStop;

#define FD922_VideoOutStop_A(n) (0x64 + n * 16)

__always_inline FD922_VideoOutStop FD922_VideoOutStop_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutStop_A(n));
	return *((FD922_VideoOutStop *)&r);
}

__always_inline void FD922_VideoOutStop_W(uint32_t *csr, unsigned int n, FD922_VideoOutStop v)
{
	__write_reg(csr, FD922_VideoOutStop_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutField {
	uint32_t switchOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t switchEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD922_VideoOutField;

#define FD922_VideoOutField_A(n) (0x65 + n * 16)

__always_inline FD922_VideoOutField FD922_VideoOutField_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutField_A(n));
	return *((FD922_VideoOutField *)&r);
}

__always_inline void FD922_VideoOutField_W(uint32_t *csr, unsigned int n, FD922_VideoOutField v)
{
	__write_reg(csr, FD922_VideoOutField_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutAudio {
	uint32_t address : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD922_VideoOutAudio;

#define FD922_VideoOutAudio_A(n) (0x66 + n * 16)

__always_inline FD922_VideoOutAudio FD922_VideoOutAudio_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutAudio_A(n));
	return *((FD922_VideoOutAudio *)&r);
}

__always_inline void FD922_VideoOutAudio_W(uint32_t *csr, unsigned int n, FD922_VideoOutAudio v)
{
	__write_reg(csr, FD922_VideoOutAudio_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutVPID {
	uint32_t vpid : 32;
} __attribute__((packed)) FD922_VideoOutVPID;

#define FD922_VideoOutVPID_A(n) (0x67 + n * 16)

__always_inline FD922_VideoOutVPID FD922_VideoOutVPID_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutVPID_A(n));
	return *((FD922_VideoOutVPID *)&r);
}

__always_inline void FD922_VideoOutVPID_W(uint32_t *csr, unsigned int n, FD922_VideoOutVPID v)
{
	__write_reg(csr, FD922_VideoOutVPID_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutOddDataCount {
	uint32_t count : 26;
	uint32_t _unused3126 : 6;
} __attribute__((packed)) FD922_VideoOutOddDataCount;

#define FD922_VideoOutOddDataCount_A(n) (0x68 + n * 16)

__always_inline FD922_VideoOutOddDataCount FD922_VideoOutOddDataCount_R(uint32_t *csr,
									unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutOddDataCount_A(n));
	return *((FD922_VideoOutOddDataCount *)&r);
}

__always_inline void FD922_VideoOutOddDataCount_W(uint32_t *csr, unsigned int n,
						  FD922_VideoOutOddDataCount v)
{
	__write_reg(csr, FD922_VideoOutOddDataCount_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutEvenDataCount {
	uint32_t count : 26;
	uint32_t _unused3126 : 6;
} __attribute__((packed)) FD922_VideoOutEvenDataCount;

#define FD922_VideoOutEvenDataCount_A(n) (0x69 + n * 16)

__always_inline FD922_VideoOutEvenDataCount FD922_VideoOutEvenDataCount_R(uint32_t *csr,
									  unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutEvenDataCount_A(n));
	return *((FD922_VideoOutEvenDataCount *)&r);
}

__always_inline void FD922_VideoOutEvenDataCount_W(uint32_t *csr, unsigned int n,
						   FD922_VideoOutEvenDataCount v)
{
	__write_reg(csr, FD922_VideoOutEvenDataCount_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutFIFOStatus {
	uint32_t videoMin : 15;
	uint32_t videoError : 1;
	uint32_t audioMin : 15;
	uint32_t audioError : 1;
} __attribute__((packed)) FD922_VideoOutFIFOStatus;

#define FD922_VideoOutFIFOStatus_A(n) (0x6A + n * 16)

__always_inline FD922_VideoOutFIFOStatus FD922_VideoOutFIFOStatus_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutFIFOStatus_A(n));
	return *((FD922_VideoOutFIFOStatus *)&r);
}

__always_inline void FD922_VideoOutFIFOStatus_W(uint32_t *csr, unsigned int n,
						FD922_VideoOutFIFOStatus v)
{
	__write_reg(csr, FD922_VideoOutFIFOStatus_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutATC {
	uint32_t data : 30;
	uint32_t address : 2;
} __attribute__((packed)) FD922_VideoOutATC;

#define FD922_VideoOutATC_A(n) (0x6B + n * 16)

__always_inline FD922_VideoOutATC FD922_VideoOutATC_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutATC_A(n));
	return *((FD922_VideoOutATC *)&r);
}

__always_inline void FD922_VideoOutATC_W(uint32_t *csr, unsigned int n, FD922_VideoOutATC v)
{
	__write_reg(csr, FD922_VideoOutATC_A(n), *((uint32_t *)&v));
}
typedef struct _FD922_VideoOutIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD922_VideoOutIRQRefTime;

#define FD922_VideoOutIRQRefTime_A(n) (0x6C + n * 16)

__always_inline FD922_VideoOutIRQRefTime FD922_VideoOutIRQRefTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutIRQRefTime_A(n));
	return *((FD922_VideoOutIRQRefTime *)&r);
}

typedef struct _FD922_VideoOutPhaseShift {
	uint32_t phase : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD922_VideoOutPhaseShift;

#define FD922_VideoOutPhaseShift_A(n) (0x6D + n * 16)

__always_inline FD922_VideoOutPhaseShift FD922_VideoOutPhaseShift_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD922_VideoOutPhaseShift_A(n));
	return *((FD922_VideoOutPhaseShift *)&r);
}

__always_inline void FD922_VideoOutPhaseShift_W(uint32_t *csr, unsigned int n,
						FD922_VideoOutPhaseShift v)
{
	__write_reg(csr, FD922_VideoOutPhaseShift_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_VideoOutWatchdog {
	uint32_t timeout : 16;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD922_VideoOutWatchdog;

#define FD922_VideoOutWatchdog_A(n) (0x6E + n * 16)

__always_inline void FD922_VideoOutWatchdog_W(uint32_t *csr, unsigned int n,
					      FD922_VideoOutWatchdog v)
{
	__write_reg(csr, FD922_VideoOutWatchdog_A(n), *((uint32_t *)&v));
}

typedef struct _FD922_GlobalCSR {
	uint32_t videoReset : 1;
	uint32_t vpllReset : 1;
	uint32_t reboot : 1;
	uint32_t _unused1503 : 13;
	uint32_t vpllLocked : 1;
	uint32_t videoClkValid : 1;
	uint32_t qpll0Locked : 1;
	uint32_t qpll1Locked : 1;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD922_GlobalCSR;

#define FD922_GlobalCSR_A (0x80)

__always_inline FD922_GlobalCSR FD922_GlobalCSR_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_GlobalCSR_A);
	return *((FD922_GlobalCSR *)&r);
}

__always_inline void FD922_GlobalCSR_W(uint32_t *csr, FD922_GlobalCSR v)
{
	__write_reg(csr, FD922_GlobalCSR_A, *((uint32_t *)&v));
}

typedef struct _FD922_MonitorStatus {
	uint32_t criticalOverheat : 1;
	uint32_t overheat : 1;
	uint32_t coreVoltageError : 1;
	uint32_t auxVoltageError : 1;
	uint32_t hisCriticalOverheat : 1;
	uint32_t hisOverheat : 1;
	uint32_t hisCoreVoltageError : 1;
	uint32_t hisAuxVoltageError : 1;
	uint32_t jtagPoison : 1;
	uint32_t _unused3109 : 23;
} __attribute__((packed)) FD922_MonitorStatus;

#define FD922_MonitorStatus_A (0x81)

__always_inline FD922_MonitorStatus FD922_MonitorStatus_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_MonitorStatus_A);
	return *((FD922_MonitorStatus *)&r);
}

__always_inline void FD922_MonitorStatus_W(uint32_t *csr, FD922_MonitorStatus v)
{
	__write_reg(csr, FD922_MonitorStatus_A, *((uint32_t *)&v));
}

typedef struct _FD922_TemperatureMonitor {
	uint32_t temperature : 10;
	uint32_t maxTemperature : 10;
	uint32_t minTemperature : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD922_TemperatureMonitor;

#define FD922_TemperatureMonitor_A (0x82)

__always_inline FD922_TemperatureMonitor FD922_TemperatureMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_TemperatureMonitor_A);
	return *((FD922_TemperatureMonitor *)&r);
}

__always_inline void FD922_TemperatureMonitor_W(uint32_t *csr, FD922_TemperatureMonitor v)
{
	__write_reg(csr, FD922_TemperatureMonitor_A, *((uint32_t *)&v));
}

typedef struct _FD922_VCoreMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD922_VCoreMonitor;

#define FD922_VCoreMonitor_A (0x83)

__always_inline FD922_VCoreMonitor FD922_VCoreMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_VCoreMonitor_A);
	return *((FD922_VCoreMonitor *)&r);
}

__always_inline void FD922_VCoreMonitor_W(uint32_t *csr, FD922_VCoreMonitor v)
{
	__write_reg(csr, FD922_VCoreMonitor_A, *((uint32_t *)&v));
}

typedef struct _FD922_VAuxMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD922_VAuxMonitor;

#define FD922_VAuxMonitor_A (0x84)

__always_inline FD922_VAuxMonitor FD922_VAuxMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_VAuxMonitor_A);
	return *((FD922_VAuxMonitor *)&r);
}

__always_inline void FD922_VAuxMonitor_W(uint32_t *csr, FD922_VAuxMonitor v)
{
	__write_reg(csr, FD922_VAuxMonitor_A, *((uint32_t *)&v));
}

typedef struct _FD922_SPIControl {
	uint32_t bits : 12;
	uint32_t read : 1;
	uint32_t write : 1;
	uint32_t ready : 1;
	uint32_t cs : 1;
	uint32_t bus : 3;
	uint32_t _unused3119 : 13;
} __attribute__((packed)) FD922_SPIControl;

#define FD922_SPIControl_A (0x85)

__always_inline FD922_SPIControl FD922_SPIControl_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_SPIControl_A);
	return *((FD922_SPIControl *)&r);
}

__always_inline void FD922_SPIControl_W(uint32_t *csr, FD922_SPIControl v)
{
	__write_reg(csr, FD922_SPIControl_A, *((uint32_t *)&v));
}

typedef struct _FD922_SPIData {
	uint32_t data : 32;
} __attribute__((packed)) FD922_SPIData;

#define FD922_SPIData_A (0x86)

__always_inline FD922_SPIData FD922_SPIData_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_SPIData_A);
	return *((FD922_SPIData *)&r);
}

__always_inline void FD922_SPIData_W(uint32_t *csr, FD922_SPIData v)
{
	__write_reg(csr, FD922_SPIData_A, *((uint32_t *)&v));
}

typedef struct _FD922_VideoClockCounter {
	uint32_t counter : 32;
} __attribute__((packed)) FD922_VideoClockCounter;

#define FD922_VideoClockCounter_A (0x89)

__always_inline FD922_VideoClockCounter FD922_VideoClockCounter_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_VideoClockCounter_A);
	return *((FD922_VideoClockCounter *)&r);
}

__always_inline void FD922_VideoClockCounter_W(uint32_t *csr, FD922_VideoClockCounter v)
{
	__write_reg(csr, FD922_VideoClockCounter_A, *((uint32_t *)&v));
}

typedef struct _FD922_ClockFrequency {
	uint32_t frequency : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD922_ClockFrequency;

#define FD922_ClockFrequency_A (0x8A)

__always_inline FD922_ClockFrequency FD922_ClockFrequency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_ClockFrequency_A);
	return *((FD922_ClockFrequency *)&r);
}

__always_inline void FD922_ClockFrequency_W(uint32_t *csr, FD922_ClockFrequency v)
{
	__write_reg(csr, FD922_ClockFrequency_A, *((uint32_t *)&v));
}

typedef struct _FD922_PCIEControlState {
	uint32_t generation : 4;
	uint32_t width : 6;
	uint32_t maxPayload : 3;
	uint32_t maxReadReq : 3;
	uint32_t tlpWriteBuffer : 6;
	uint32_t tlpMaxRead : 6;
	uint32_t wrNoSnoop : 1;
	uint32_t rdNoSnoop : 1;
	uint32_t tlpNoSnoop : 1;
	uint32_t _unused3131 : 1;
} __attribute__((packed)) FD922_PCIEControlState;

#define FD922_PCIEControlState_A (0x8D)

__always_inline FD922_PCIEControlState FD922_PCIEControlState_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_PCIEControlState_A);
	return *((FD922_PCIEControlState *)&r);
}

__always_inline void FD922_PCIEControlState_W(uint32_t *csr, FD922_PCIEControlState v)
{
	__write_reg(csr, FD922_PCIEControlState_A, *((uint32_t *)&v));
}

typedef struct _FD922_PCIEErrors {
	uint32_t pcieErrors : 15;
	uint32_t pcieErrorsReset : 1;
	uint32_t pcieUnknown : 15;
	uint32_t pcieUnknownReset : 1;
} __attribute__((packed)) FD922_PCIEErrors;

#define FD922_PCIEErrors_A (0x8E)

__always_inline FD922_PCIEErrors FD922_PCIEErrors_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_PCIEErrors_A);
	return *((FD922_PCIEErrors *)&r);
}

__always_inline void FD922_PCIEErrors_W(uint32_t *csr, FD922_PCIEErrors v)
{
	__write_reg(csr, FD922_PCIEErrors_A, *((uint32_t *)&v));
}

typedef struct _FD922_SoftID {
	uint32_t id : 32;
} __attribute__((packed)) FD922_SoftID;

#define FD922_SoftID_A (0x8F)

__always_inline FD922_SoftID FD922_SoftID_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_SoftID_A);
	return *((FD922_SoftID *)&r);
}

__always_inline void FD922_SoftID_W(uint32_t *csr, FD922_SoftID v)
{
	__write_reg(csr, FD922_SoftID_A, *((uint32_t *)&v));
}

typedef struct _FD922_PCIEPerfomance {
	uint32_t txCounter : 16;
	uint32_t rxCounter : 16;
} __attribute__((packed)) FD922_PCIEPerfomance;

#define FD922_PCIEPerfomance_A (0x93)

__always_inline FD922_PCIEPerfomance FD922_PCIEPerfomance_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_PCIEPerfomance_A);
	return *((FD922_PCIEPerfomance *)&r);
}

__always_inline void FD922_PCIEPerfomance_W(uint32_t *csr, FD922_PCIEPerfomance v)
{
	__write_reg(csr, FD922_PCIEPerfomance_A, *((uint32_t *)&v));
}

typedef struct _FD922_PCIERxMaxLatency {
	uint32_t maxLatency : 16;
	uint32_t numRequests : 16;
} __attribute__((packed)) FD922_PCIERxMaxLatency;

#define FD922_PCIERxMaxLatency_A (0x94)

__always_inline FD922_PCIERxMaxLatency FD922_PCIERxMaxLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_PCIERxMaxLatency_A);
	return *((FD922_PCIERxMaxLatency *)&r);
}

__always_inline void FD922_PCIERxMaxLatency_W(uint32_t *csr, FD922_PCIERxMaxLatency v)
{
	__write_reg(csr, FD922_PCIERxMaxLatency_A, *((uint32_t *)&v));
}

typedef struct _FD922_PCIERxTotalLatency {
	uint32_t latency : 32;
} __attribute__((packed)) FD922_PCIERxTotalLatency;

#define FD922_PCIERxTotalLatency_A (0x95)

__always_inline FD922_PCIERxTotalLatency FD922_PCIERxTotalLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_PCIERxTotalLatency_A);
	return *((FD922_PCIERxTotalLatency *)&r);
}

__always_inline void FD922_PCIERxTotalLatency_W(uint32_t *csr, FD922_PCIERxTotalLatency v)
{
	__write_reg(csr, FD922_PCIERxTotalLatency_A, *((uint32_t *)&v));
}

typedef struct _FD922_StaticVersion {
	uint32_t revision : 16;
	uint32_t major : 8;
	uint32_t hardware : 4;
	uint32_t project : 4;
} __attribute__((packed)) FD922_StaticVersion;

#define FD922_StaticVersion_A (0xFF)

__always_inline FD922_StaticVersion FD922_StaticVersion_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD922_StaticVersion_A);
	return *((FD922_StaticVersion *)&r);
}

__always_inline void FD922_StaticVersion_W(uint32_t *csr, FD922_StaticVersion v)
{
	__write_reg(csr, FD922_StaticVersion_A, *((uint32_t *)&v));
}

#endif
