/*
   fd940_reg.h - register definitions driver for SoftLab-NSK FD940 board

   Copyright (C) 2017 - 2024 SoftLab-NSK <forward@softlab.tv>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef FD940_REG_H
#define FD940_REG_H

#include <linux/types.h>
#include <linux/io.h>

#define __read_reg(csr, addr) ioread32(&csr[addr])
#define __write_reg(csr, addr, v) iowrite32(v, &csr[addr])

typedef struct _FD940_VDMA {
	uint32_t _unused3000 : 31;
	uint32_t enable : 1;
} __attribute__((packed)) FD940_VDMA;

#define FD940_VDMA_A (0x00)

__always_inline FD940_VDMA FD940_VDMA_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_VDMA_A);
	return *((FD940_VDMA *)&r);
}

__always_inline void FD940_VDMA_W(uint32_t *csr, FD940_VDMA v)
{
	__write_reg(csr, FD940_VDMA_A, *((uint32_t *)&v));
}

typedef struct _FD940_VDMADescriptor {
	uint32_t address : 32;
} __attribute__((packed)) FD940_VDMADescriptor;

#define FD940_VDMADescriptor_A (0x01)

__always_inline FD940_VDMADescriptor FD940_VDMADescriptor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_VDMADescriptor_A);
	return *((FD940_VDMADescriptor *)&r);
}

__always_inline void FD940_VDMADescriptor_W(uint32_t *csr, FD940_VDMADescriptor v)
{
	__write_reg(csr, FD940_VDMADescriptor_A, *((uint32_t *)&v));
}

typedef struct _FD940_IRQEnable {
	uint32_t in0IRQ : 1;
	uint32_t _unused0201 : 2;
	uint32_t in1IRQ : 1;
	uint32_t _unused0504 : 2;
	uint32_t in2IRQ : 1;
	uint32_t _unused0807 : 2;
	uint32_t in3IRQ : 1;
	uint32_t _unused1110 : 2;
	uint32_t sync0IRQ : 1;
	uint32_t _unused1313 : 1;
	uint32_t sync1IRQ : 1;
	uint32_t _unused1515 : 1;
	uint32_t sync2IRQ : 1;
	uint32_t _unused1717 : 1;
	uint32_t sync3IRQ : 1;
	uint32_t _unused3119 : 13;
} __attribute__((packed)) FD940_IRQEnable;

#define FD940_IRQEnable_A (0x04)

__always_inline FD940_IRQEnable FD940_IRQEnable_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_IRQEnable_A);
	return *((FD940_IRQEnable *)&r);
}

__always_inline void FD940_IRQEnable_W(uint32_t *csr, FD940_IRQEnable v)
{
	__write_reg(csr, FD940_IRQEnable_A, *((uint32_t *)&v));
}

typedef struct _FD940_IRQFlags {
	uint32_t in0IRQ : 1;
	uint32_t in0Buffer : 2;
	uint32_t in1IRQ : 1;
	uint32_t in1Buffer : 2;
	uint32_t in2IRQ : 1;
	uint32_t in2Buffer : 2;
	uint32_t in3IRQ : 1;
	uint32_t in3Buffer : 2;
	uint32_t sync0IRQ : 1;
	uint32_t sync0Field : 1;
	uint32_t sync1IRQ : 1;
	uint32_t sync1Field : 1;
	uint32_t sync2IRQ : 1;
	uint32_t sync2Field : 1;
	uint32_t sync3IRQ : 1;
	uint32_t sync3Field : 1;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD940_IRQFlags;

#define FD940_IRQFlags_A (0x05)

__always_inline FD940_IRQFlags FD940_IRQFlags_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_IRQFlags_A);
	return *((FD940_IRQFlags *)&r);
}

__always_inline void FD940_IRQFlags_W(uint32_t *csr, FD940_IRQFlags v)
{
	__write_reg(csr, FD940_IRQFlags_A, *((uint32_t *)&v));
}

typedef struct _FD940_SyncGenerator {
	uint32_t frameSize : 23;
	uint32_t _unused3023 : 8;
	uint32_t sync : 1;
} __attribute__((packed)) FD940_SyncGenerator;

#define FD940_SyncGenerator_A(n) (0x08 + n * 4)

__always_inline FD940_SyncGenerator FD940_SyncGenerator_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_SyncGenerator_A(n));
	return *((FD940_SyncGenerator *)&r);
}

__always_inline void FD940_SyncGenerator_W(uint32_t *csr, unsigned int n, FD940_SyncGenerator v)
{
	__write_reg(csr, FD940_SyncGenerator_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_SyncGeneratorField {
	uint32_t fieldSize : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD940_SyncGeneratorField;

#define FD940_SyncGeneratorField_A(n) (0x09 + n * 4)

__always_inline FD940_SyncGeneratorField FD940_SyncGeneratorField_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_SyncGeneratorField_A(n));
	return *((FD940_SyncGeneratorField *)&r);
}

__always_inline void FD940_SyncGeneratorField_W(uint32_t *csr, unsigned int n,
						FD940_SyncGeneratorField v)
{
	__write_reg(csr, FD940_SyncGeneratorField_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_SyncGeneratorTime {
	uint32_t time : 32;
} __attribute__((packed)) FD940_SyncGeneratorTime;

#define FD940_SyncGeneratorTime_A(n) (0x0A + n * 4)

__always_inline FD940_SyncGeneratorTime FD940_SyncGeneratorTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_SyncGeneratorTime_A(n));
	return *((FD940_SyncGeneratorTime *)&r);
}

__always_inline void FD940_SyncGeneratorTime_W(uint32_t *csr, unsigned int n,
					       FD940_SyncGeneratorTime v)
{
	__write_reg(csr, FD940_SyncGeneratorTime_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_SyncGeneratorPhase {
	uint32_t phase : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD940_SyncGeneratorPhase;

#define FD940_SyncGeneratorPhase_A(n) (0x0B + n * 4)

__always_inline FD940_SyncGeneratorPhase FD940_SyncGeneratorPhase_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_SyncGeneratorPhase_A(n));
	return *((FD940_SyncGeneratorPhase *)&r);
}

__always_inline void FD940_SyncGeneratorPhase_W(uint32_t *csr, unsigned int n,
						FD940_SyncGeneratorPhase v)
{
	__write_reg(csr, FD940_SyncGeneratorPhase_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_VideoInCS {
	uint32_t enable : 1;
	uint32_t hpd : 1;
	uint32_t rateLocked : 1;
	uint32_t rate : 3;
	uint32_t formatLocked : 1;
	uint32_t progressive : 1;
	uint32_t capture : 1;
	uint32_t captureColor : 2;
	uint32_t captureDepth : 2;
	uint32_t pixrep : 1;
	uint32_t dvi : 1;
	uint32_t edidOverride : 1;
	uint32_t bitrate : 15;
	uint32_t test : 1;
} __attribute__((packed)) FD940_VideoInCS;

#define FD940_VideoInCS_A(n) (0x40 + n * 16)

__always_inline FD940_VideoInCS FD940_VideoInCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_VideoInCS_A(n));
	return *((FD940_VideoInCS *)&r);
}

__always_inline void FD940_VideoInCS_W(uint32_t *csr, unsigned int n, FD940_VideoInCS v)
{
	__write_reg(csr, FD940_VideoInCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_VideoInLine {
	uint32_t totalLines : 12;
	uint32_t _unused1512 : 4;
	uint32_t activeLines : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD940_VideoInLine;

#define FD940_VideoInLine_A(n) (0x41 + n * 16)

__always_inline FD940_VideoInLine FD940_VideoInLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_VideoInLine_A(n));
	return *((FD940_VideoInLine *)&r);
}

__always_inline void FD940_VideoInLine_W(uint32_t *csr, unsigned int n, FD940_VideoInLine v)
{
	__write_reg(csr, FD940_VideoInLine_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_VideoInPixel {
	uint32_t totalPixels : 14;
	uint32_t _unused1514 : 2;
	uint32_t activePixels : 14;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD940_VideoInPixel;

#define FD940_VideoInPixel_A(n) (0x42 + n * 16)

__always_inline FD940_VideoInPixel FD940_VideoInPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_VideoInPixel_A(n));
	return *((FD940_VideoInPixel *)&r);
}

__always_inline void FD940_VideoInPixel_W(uint32_t *csr, unsigned int n, FD940_VideoInPixel v)
{
	__write_reg(csr, FD940_VideoInPixel_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_VideoInIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD940_VideoInIRQRefTime;

#define FD940_VideoInIRQRefTime_A(n) (0x44 + n * 16)

__always_inline FD940_VideoInIRQRefTime FD940_VideoInIRQRefTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_VideoInIRQRefTime_A(n));
	return *((FD940_VideoInIRQRefTime *)&r);
}

__always_inline void FD940_VideoInIRQRefTime_W(uint32_t *csr, unsigned int n,
					       FD940_VideoInIRQRefTime v)
{
	__write_reg(csr, FD940_VideoInIRQRefTime_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_VideoInANCCounter {
	uint32_t counter : 21;
	uint32_t _unused3121 : 11;
} __attribute__((packed)) FD940_VideoInANCCounter;

#define FD940_VideoInANCCounter_A(n) (0x46 + n * 16)

__always_inline FD940_VideoInANCCounter FD940_VideoInANCCounter_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_VideoInANCCounter_A(n));
	return *((FD940_VideoInANCCounter *)&r);
}

__always_inline void FD940_VideoInANCCounter_W(uint32_t *csr, unsigned int n,
					       FD940_VideoInANCCounter v)
{
	__write_reg(csr, FD940_VideoInANCCounter_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_VideoInInfoFrame {
	uint32_t aviVIC : 8;
	uint32_t aviY : 3;
	uint32_t aviYQ : 2;
	uint32_t aviPR : 4;
	uint32_t aviValid : 1;
	uint32_t aiCT : 4;
	uint32_t aiCC : 3;
	uint32_t aiSF : 3;
	uint32_t aiSS : 2;
	uint32_t aiValid : 1;
	uint32_t _unused3131 : 1;
} __attribute__((packed)) FD940_VideoInInfoFrame;

#define FD940_VideoInInfoFrame_A(n) (0x47 + n * 16)

__always_inline FD940_VideoInInfoFrame FD940_VideoInInfoFrame_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_VideoInInfoFrame_A(n));
	return *((FD940_VideoInInfoFrame *)&r);
}

__always_inline void FD940_VideoInInfoFrame_W(uint32_t *csr, unsigned int n,
					      FD940_VideoInInfoFrame v)
{
	__write_reg(csr, FD940_VideoInInfoFrame_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_VideoInFIFOStatus {
	uint32_t videoMax : 15;
	uint32_t videoError : 1;
	uint32_t audioMax : 15;
	uint32_t audioError : 1;
} __attribute__((packed)) FD940_VideoInFIFOStatus;

#define FD940_VideoInFIFOStatus_A(n) (0x4A + n * 16)

__always_inline FD940_VideoInFIFOStatus FD940_VideoInFIFOStatus_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_VideoInFIFOStatus_A(n));
	return *((FD940_VideoInFIFOStatus *)&r);
}

__always_inline void FD940_VideoInFIFOStatus_W(uint32_t *csr, unsigned int n,
					       FD940_VideoInFIFOStatus v)
{
	__write_reg(csr, FD940_VideoInFIFOStatus_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_VideoInColorConversion {
	uint32_t data : 16;
	uint32_t address : 4;
	uint32_t write : 1;
	uint32_t clamp : 1;
	uint32_t clampRGB : 1;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD940_VideoInColorConversion;

#define FD940_VideoInColorConversion_A(n) (0x4C + n * 16)

__always_inline FD940_VideoInColorConversion FD940_VideoInColorConversion_R(uint32_t *csr,
									    unsigned int n)
{
	uint32_t r = __read_reg(csr, FD940_VideoInColorConversion_A(n));
	return *((FD940_VideoInColorConversion *)&r);
}

__always_inline void FD940_VideoInColorConversion_W(uint32_t *csr, unsigned int n,
						    FD940_VideoInColorConversion v)
{
	__write_reg(csr, FD940_VideoInColorConversion_A(n), *((uint32_t *)&v));
}

typedef struct _FD940_GlobalCSR {
	uint32_t videoReset : 1;
	uint32_t _unused3101 : 31;
} __attribute__((packed)) FD940_GlobalCSR;

#define FD940_GlobalCSR_A (0x80)

__always_inline FD940_GlobalCSR FD940_GlobalCSR_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_GlobalCSR_A);
	return *((FD940_GlobalCSR *)&r);
}

__always_inline void FD940_GlobalCSR_W(uint32_t *csr, FD940_GlobalCSR v)
{
	__write_reg(csr, FD940_GlobalCSR_A, *((uint32_t *)&v));
}

typedef struct _FD940_MonitorStatus {
	uint32_t criticalOverheat : 1;
	uint32_t overheat : 1;
	uint32_t coreVoltageError : 1;
	uint32_t auxVoltageError : 1;
	uint32_t hisCriticalOverheat : 1;
	uint32_t hisOverheat : 1;
	uint32_t hisCoreVoltageError : 1;
	uint32_t hisAuxVoltageError : 1;
	uint32_t jtagPoison : 1;
	uint32_t _unused3109 : 23;
} __attribute__((packed)) FD940_MonitorStatus;

#define FD940_MonitorStatus_A (0x81)

__always_inline FD940_MonitorStatus FD940_MonitorStatus_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_MonitorStatus_A);
	return *((FD940_MonitorStatus *)&r);
}

__always_inline void FD940_MonitorStatus_W(uint32_t *csr, FD940_MonitorStatus v)
{
	__write_reg(csr, FD940_MonitorStatus_A, *((uint32_t *)&v));
}

typedef struct _FD940_TemperatureMonitor {
	uint32_t temperature : 10;
	uint32_t maxTemperature : 10;
	uint32_t minTemperature : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD940_TemperatureMonitor;

#define FD940_TemperatureMonitor_A (0x82)

__always_inline FD940_TemperatureMonitor FD940_TemperatureMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_TemperatureMonitor_A);
	return *((FD940_TemperatureMonitor *)&r);
}

__always_inline void FD940_TemperatureMonitor_W(uint32_t *csr, FD940_TemperatureMonitor v)
{
	__write_reg(csr, FD940_TemperatureMonitor_A, *((uint32_t *)&v));
}

typedef struct _FD940_VCoreMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD940_VCoreMonitor;

#define FD940_VCoreMonitor_A (0x83)

__always_inline FD940_VCoreMonitor FD940_VCoreMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_VCoreMonitor_A);
	return *((FD940_VCoreMonitor *)&r);
}

__always_inline void FD940_VCoreMonitor_W(uint32_t *csr, FD940_VCoreMonitor v)
{
	__write_reg(csr, FD940_VCoreMonitor_A, *((uint32_t *)&v));
}

typedef struct _FD940_VAuxMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD940_VAuxMonitor;

#define FD940_VAuxMonitor_A (0x84)

__always_inline FD940_VAuxMonitor FD940_VAuxMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_VAuxMonitor_A);
	return *((FD940_VAuxMonitor *)&r);
}

__always_inline void FD940_VAuxMonitor_W(uint32_t *csr, FD940_VAuxMonitor v)
{
	__write_reg(csr, FD940_VAuxMonitor_A, *((uint32_t *)&v));
}

typedef struct _FD940_VideoClockCounter {
	uint32_t counter : 32;
} __attribute__((packed)) FD940_VideoClockCounter;

#define FD940_VideoClockCounter_A (0x89)

__always_inline FD940_VideoClockCounter FD940_VideoClockCounter_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_VideoClockCounter_A);
	return *((FD940_VideoClockCounter *)&r);
}

__always_inline void FD940_VideoClockCounter_W(uint32_t *csr, FD940_VideoClockCounter v)
{
	__write_reg(csr, FD940_VideoClockCounter_A, *((uint32_t *)&v));
}

typedef struct _FD940_SoftID {
	uint32_t id : 32;
} __attribute__((packed)) FD940_SoftID;

#define FD940_SoftID_A (0x8F)

__always_inline FD940_SoftID FD940_SoftID_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_SoftID_A);
	return *((FD940_SoftID *)&r);
}

__always_inline void FD940_SoftID_W(uint32_t *csr, FD940_SoftID v)
{
	__write_reg(csr, FD940_SoftID_A, *((uint32_t *)&v));
}

typedef struct _FD940_HardID0 {
	uint32_t id : 32;
} __attribute__((packed)) FD940_HardID0;

#define FD940_HardID0_A (0x90)

__always_inline FD940_HardID0 FD940_HardID0_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_HardID0_A);
	return *((FD940_HardID0 *)&r);
}

__always_inline void FD940_HardID0_W(uint32_t *csr, FD940_HardID0 v)
{
	__write_reg(csr, FD940_HardID0_A, *((uint32_t *)&v));
}

typedef struct _FD940_HardID1 {
	uint32_t id : 32;
} __attribute__((packed)) FD940_HardID1;

#define FD940_HardID1_A (0x91)

__always_inline FD940_HardID1 FD940_HardID1_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_HardID1_A);
	return *((FD940_HardID1 *)&r);
}

__always_inline void FD940_HardID1_W(uint32_t *csr, FD940_HardID1 v)
{
	__write_reg(csr, FD940_HardID1_A, *((uint32_t *)&v));
}

typedef struct _FD940_HardID2 {
	uint32_t id : 32;
} __attribute__((packed)) FD940_HardID2;

#define FD940_HardID2_A (0x92)

__always_inline FD940_HardID2 FD940_HardID2_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_HardID2_A);
	return *((FD940_HardID2 *)&r);
}

__always_inline void FD940_HardID2_W(uint32_t *csr, FD940_HardID2 v)
{
	__write_reg(csr, FD940_HardID2_A, *((uint32_t *)&v));
}

typedef struct _FD940_PCIEPerfomance {
	uint32_t txCounter : 16;
	uint32_t rxCounter : 16;
} __attribute__((packed)) FD940_PCIEPerfomance;

#define FD940_PCIEPerfomance_A (0x93)

__always_inline FD940_PCIEPerfomance FD940_PCIEPerfomance_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_PCIEPerfomance_A);
	return *((FD940_PCIEPerfomance *)&r);
}

__always_inline void FD940_PCIEPerfomance_W(uint32_t *csr, FD940_PCIEPerfomance v)
{
	__write_reg(csr, FD940_PCIEPerfomance_A, *((uint32_t *)&v));
}

typedef struct _FD940_PCIERxMaxLatency {
	uint32_t maxLatency : 16;
	uint32_t numRequests : 16;
} __attribute__((packed)) FD940_PCIERxMaxLatency;

#define FD940_PCIERxMaxLatency_A (0x94)

__always_inline FD940_PCIERxMaxLatency FD940_PCIERxMaxLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_PCIERxMaxLatency_A);
	return *((FD940_PCIERxMaxLatency *)&r);
}

__always_inline void FD940_PCIERxMaxLatency_W(uint32_t *csr, FD940_PCIERxMaxLatency v)
{
	__write_reg(csr, FD940_PCIERxMaxLatency_A, *((uint32_t *)&v));
}

typedef struct _FD940_PCIERxTotalLatency {
	uint32_t latency : 32;
} __attribute__((packed)) FD940_PCIERxTotalLatency;

#define FD940_PCIERxTotalLatency_A (0x95)

__always_inline FD940_PCIERxTotalLatency FD940_PCIERxTotalLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_PCIERxTotalLatency_A);
	return *((FD940_PCIERxTotalLatency *)&r);
}

__always_inline void FD940_PCIERxTotalLatency_W(uint32_t *csr, FD940_PCIERxTotalLatency v)
{
	__write_reg(csr, FD940_PCIERxTotalLatency_A, *((uint32_t *)&v));
}

typedef struct _FD940_StaticVersion {
	uint32_t revision : 16;
	uint32_t major : 8;
	uint32_t hardware : 4;
	uint32_t project : 4;
} __attribute__((packed)) FD940_StaticVersion;

#define FD940_StaticVersion_A (0xFF)

__always_inline FD940_StaticVersion FD940_StaticVersion_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD940_StaticVersion_A);
	return *((FD940_StaticVersion *)&r);
}

__always_inline void FD940_StaticVersion_W(uint32_t *csr, FD940_StaticVersion v)
{
	__write_reg(csr, FD940_StaticVersion_A, *((uint32_t *)&v));
}

#endif
