#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <iomanip>
#include <iostream>
#include <thread>

/*!
 * \dir ./
 * This example shows how to work with PPS receiver on analog input and tune frequency on board
 */

static const int MEASURE_PERIOD = 5;

void measurePPS(V4L2Device& dev, float& avg_freq_diff)
{
    struct v4l2_ext_control ex_ctl;
    struct v4l2_ext_controls ex_ctls;
    struct v4l2_forward_analog_rx_timestamp ts;

    ex_ctl.id = V4L2_CID_FORWARD_ANALOG_RX_TIMESTAMP;
    ex_ctl.size = sizeof(struct v4l2_forward_analog_rx_timestamp);
    ex_ctl.ptr = &ts;

    ex_ctls.ctrl_class = 0;
    ex_ctls.which = V4L2_CTRL_WHICH_CUR_VAL;
    ex_ctls.count = 1;
    ex_ctls.controls = &ex_ctl;

    uint32_t prev_timestamp;
    int period_diff_acc = 0;
    int period_diff_count = 0;

    for (int i = 0; i < MEASURE_PERIOD * 2; i++) {
        if (dev.ioctl(VIDIOC_G_EXT_CTRLS, &ex_ctls)) {
            std::cerr << "Cannot get analog input PPS: " << dev.errorString() << std::endl;
            break;
        }

        if (!ts.valid)
            continue;

        if (i == 0)
            prev_timestamp = ts.timestamp;

        if (ts.timestamp != prev_timestamp) {
            int period_diff = (ts.timestamp - prev_timestamp) - 148500000;
            float freq_diff = period_diff / 148500000.0;
            prev_timestamp = ts.timestamp;

            period_diff_acc += period_diff;
            period_diff_count++;

            std::cout << "PPS: hw_timestamp = " << ts.timestamp << ", hw_time = " << ts.cur_hw_time
                      << ", sys_time = " << ts.cur_sys_time
                      << ", hw_frequency = " << freq_diff * 1000000.0 << " ppm" << std::endl;
        }

        std::this_thread::sleep_for(std::chrono::milliseconds(500));
    }

    avg_freq_diff = period_diff_acc / 148500000.0 / period_diff_count;
    std::cout << "Average frequency drift: " << avg_freq_diff * 1000000.0 << " ppm" << std::endl;
}

void tuneFrequency(V4L2Device& dev, float freq_diff_ratio)
{
    struct v4l2_ext_control ex_ctl;
    struct v4l2_ext_controls ex_ctls;
    int64_t ppt = freq_diff_ratio * 1e12;

    ex_ctl.id = V4L2_CID_FORWARD_FREQUENCY_ADJUST;
    ex_ctl.size = 0;

    ex_ctls.ctrl_class = 0;
    ex_ctls.which = V4L2_CTRL_WHICH_CUR_VAL;
    ex_ctls.count = 1;
    ex_ctls.controls = &ex_ctl;

    if (dev.ioctl(VIDIOC_G_EXT_CTRLS, &ex_ctls)) {
        std::cerr << "Cannot get frequency: " << dev.errorString() << std::endl;
        return;
    }

    std::cout << ex_ctl.value64 << std::endl;

    ex_ctl.value64 = ex_ctl.value64 + ppt;

    if (dev.ioctl(VIDIOC_S_EXT_CTRLS, &ex_ctls)) {
        std::cerr << "Cannot set frequency: " << dev.errorString() << std::endl;
        return;
    }
}

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc > 1) {
        devPath = argv[1];
    } else {
        std::vector<Board::Info> boards = Board::enumerate();
        for (auto board : boards) {
            if (board.videoDevs.size() && !board.videoDevs[0].empty())
                devPath = board.videoDevs[0];
        }
        if (devPath.empty()) {
            std::cerr << "Cannot find any device!" << std::endl;
            return -1;
        }
    }

    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    struct v4l2_control ctl;
    ctl.id = V4L2_CID_FORWARD_ANALOG_RX_MODE;
    ctl.value = V4L_FORWARD_ANALOG_RX_MODE_PPS;

    if (dev.ioctl(VIDIOC_S_CTRL, &ctl)) {
        std::cerr << "Cannot enable PPS receiver: " << dev.errorString() << std::endl;
        return -1;
    }

    float freq_diff;
    measurePPS(dev, freq_diff);
    tuneFrequency(dev, -freq_diff);
    measurePPS(dev, freq_diff);

    return 0;
}
