#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <poll.h>
#include <thread>

/*!
 * \dir ./
 * This example shows how to work with VITC/LTC receiver on analog input
 */

static const bool LTC = false; // Receive LTC (true) or VITC + genlock (false)
static bool shouldStop = false;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

std::ostream& operator<<(std::ostream& str, const struct v4l2_timecode& tc)
{
    char flags[6];
    flags[0] = (tc.flags & V4L2_TC_FLAG_DROPFRAME) ? 'D' : '-';
    flags[1] = (tc.flags & V4L2_TC_FLAG_COLORFRAME) ? 'C' : '-';
    flags[2] = (tc.flags & (1 << 2)) ? '1' : '0';
    flags[3] = (tc.flags & (1 << 3)) ? '1' : '0';
    flags[4] = (tc.flags & (1 << 4)) ? '1' : '0';
    flags[5] = '\0';

    std::string fps;
    switch (tc.type) {
    case V4L2_TC_TYPE_24FPS:
        fps = "24Hz";
        break;
    case V4L2_TC_TYPE_25FPS:
        fps = "25Hz";
        break;
    case V4L2_TC_TYPE_30FPS:
        fps = "30Hz";
        break;
    case V4L2_TC_TYPE_50FPS:
        fps = "50Hz";
        break;
    case V4L2_TC_TYPE_60FPS:
        fps = "60Hz";
        break;
    default:
        fps = "    ";
        break;
    }

    str << std::dec;
    str << std::setw(2) << std::setfill('0') << (int)tc.hours;
    str << ":" << std::setw(2) << std::setfill('0') << (int)tc.minutes;
    str << ":" << std::setw(2) << std::setfill('0') << (int)tc.seconds;
    str << "." << std::setw(2) << std::setfill('0') << (int)tc.frames;
    str << " " << flags;
    str << " " << std::setw(2) << std::setfill('0') << std::hex << (int)tc.userbits[3]
        << (int)tc.userbits[2] << (int)tc.userbits[1] << (int)tc.userbits[0];
    str << " (" << fps << ")";

    return str;
}

bool operator==(const struct v4l2_timecode& x1, const struct v4l2_timecode& x2)
{
    return (x1.type == x2.type) && (x1.flags == x2.flags) && (x1.frames == x2.frames)
        && (x1.seconds == x2.seconds) && (x1.minutes == x2.minutes) && (x1.hours == x2.hours);
}

int getTimeCode(V4L2Device& dev, struct v4l2_timecode* tc)
{
    struct v4l2_ext_control ex_ctl;
    struct v4l2_ext_controls ex_ctls;

    ex_ctl.id = V4L2_CID_FORWARD_ANALOG_RX_TIMECODE;
    ex_ctl.size = sizeof(struct v4l2_timecode);
    ex_ctl.ptr = tc;

    ex_ctls.ctrl_class = 0;
    ex_ctls.which = V4L2_CTRL_WHICH_CUR_VAL;
    ex_ctls.count = 1;
    ex_ctls.controls = &ex_ctl;

    return dev.ioctl(VIDIOC_G_EXT_CTRLS, &ex_ctls);
}

int enableGenlock(V4L2Device& dev)
{
    struct v4l2_control ctl;

    std::cout << "Enabling genlock on analog input ... ";
    std::cout.flush();

    ctl.id = V4L2_CID_FORWARD_ANALOG_RX_MODE;
    ctl.value = V4L_FORWARD_ANALOG_RX_MODE_GENLOCK;
    dev.ioctl(VIDIOC_S_CTRL, &ctl);

    ctl.id = V4L2_CID_FORWARD_GENLOCK_SOURCE;
    ctl.value = V4L_FORWARD_GENLOCK_SRC_ANALOG;
    dev.ioctl(VIDIOC_S_CTRL, &ctl);

    int timeout = 20;
    do {
        std::this_thread::sleep_for(std::chrono::milliseconds(500));
        ctl.id = V4L2_CID_FORWARD_GENLOCK_STATE;
        dev.ioctl(VIDIOC_G_CTRL, &ctl);

        if ((ctl.value == V4L_FORWARD_GENLOCK_NO_INPUT_SIGNAL)) {
            std::cout << "no genlock signal" << std::endl;
            return -ENOLINK;
        } else if (timeout == 0) {
            std::cout << "timeout" << std::endl;
            return -ETIMEDOUT;
        }
        timeout--;
    } while (ctl.value != V4L_FORWARD_GENLOCK_LOCKED);

    std::cout << "done" << std::endl;

    return 0;
}

void receiveVITC(V4L2Device& dev)
{
    // Enable analog genlock on output, so device will be in-sync with VITC
    enableGenlock(dev);

    v4l2_event_subscription evts;
    memset(&evts, 0, sizeof(evts));
    evts.type = V4L2_EVENT_VSYNC;

    // Subscribe to frame event
    dev.ioctl(VIDIOC_SUBSCRIBE_EVENT, &evts);

    pollfd pollfd;
    pollfd.fd = dev.fd();
    pollfd.events = POLLPRI;

    int frame = 0;
    while (!shouldStop) {
        // Wait for frame border
        poll(&pollfd, 1, 100);

        if (pollfd.revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        if (!(pollfd.revents & POLLPRI))
            continue;

        struct v4l2_event event = { 0 };
        dev.ioctl(VIDIOC_DQEVENT, &event);

        if (event.type != V4L2_EVENT_VSYNC)
            continue;

        if ((event.u.vsync.field != V4L2_FIELD_TOP) && (event.u.vsync.field != V4L2_FIELD_NONE))
            continue;

        // Get received VITC
        struct v4l2_timecode tc;
        int result = getTimeCode(dev, &tc);

        std::cout << "Frame " << std::setw(4) << std::dec << frame << ": ";
        frame++;

        if (result && (errno == ENODATA)) {
            std::cout << "no timecode" << std::endl;
            continue;
        } else if (result) {
            std::cout << "receive error: " << errno << std::endl;
            continue;
        }

        std::cout << tc << std::endl;
    }

    dev.ioctl(VIDIOC_UNSUBSCRIBE_EVENT, &evts);
}

void receiveLTC(V4L2Device& dev)
{
    struct v4l2_timecode tcp;

    struct v4l2_control ctl;
    ctl.id = V4L2_CID_FORWARD_ANALOG_RX_MODE;
    ctl.value = V4L_FORWARD_ANALOG_RX_MODE_LTC;
    dev.ioctl(VIDIOC_S_CTRL, &ctl);

    while (!shouldStop) {
        std::this_thread::sleep_for(std::chrono::milliseconds(4));

        struct v4l2_timecode tc;
        int result = getTimeCode(dev, &tc);

        if (result)
            continue;

        if (tcp == tc)
            continue;

        std::cout << "New LTC: " << tc << std::endl;

        tcp = tc;
    }
}

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc > 1) {
        devPath = argv[1];
    } else {
        std::vector<Board::Info> boards = Board::enumerate();
        for (auto board : boards) {
            if (board.videoDevs.size() && !board.videoDevs[0].empty())
                devPath = board.videoDevs[0];
        }
        if (devPath.empty()) {
            std::cerr << "Cannot find any device!" << std::endl;
            return -1;
        }
    }

    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    if (!LTC)
        receiveVITC(dev);
    else
        receiveLTC(dev);

    return 0;
}
