#include "ALSADevice.hpp"
#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <climits>
#include <cmath>
#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <list>
#include <poll.h>
#include <tuple>

/*!
 * \dir ./
 * This example generates A-V delay test pattern
 */

namespace {
static const int BUFFER_COUNT = 8;
static const int TEST_PERIOD_FIELDS = 10;
// ALSA chunk size, should be less than frame (field) period
static const int AUDIO_PERIOD_SIZE_US = 10000;
static const int AUDIO_BUFFER_SIZE_US = 1000000;
static const int AUDIO_CHANNELS = 2;
static const snd_pcm_uframes_t AUDIO_SAMPLE_RATE = 48000;
}

static bool shouldStop = false;
static uint8_t* videoTest;
snd_pcm_uframes_t audioBufSize;
int32_t* audioBufSilence;
int32_t* audioBufSin;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

//! Get required audio samples per video frame
snd_pcm_uframes_t getAudioPerVideoFrame(V4L2Device& dev, const v4l2_pix_format_mplane& fmt)
{
    v4l2_dv_timings tim;
    if (dev.ioctl(VIDIOC_G_DV_TIMINGS, &tim))
        return 0;

    uint64_t denom = tim.bt.pixelclock;
    uint64_t num
        = AUDIO_SAMPLE_RATE * V4L2_DV_BT_FRAME_WIDTH(&tim.bt) * V4L2_DV_BT_FRAME_HEIGHT(&tim.bt);

    // FIXME: No /1.001 compensation
    if (tim.bt.interlaced && (fmt.field == V4L2_FIELD_ALTERNATE))
        denom *= 2;

    return num / denom;
}

v4l2_format setupVideoFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;
    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;
    // 8-bit YUYV
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_YUYV;
    // 10-bit v210
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_V210;
    // Raw - whole SDI frame, with HANC, VANC, 10-bit dense packed
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Playback frame format is " << fmt << std::endl;

    return fmt;
}

void preroll(
    V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers, v4l2_pix_format_mplane fmt)
{
    // Fill playback queue
    for (auto buf : buffers) {
        if (fmt.field == V4L2_FIELD_ALTERNATE)
            buf->v4l2buf().field = (buf->v4l2buf().index % 2) ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;
        else
            buf->v4l2buf().field = fmt.field;

        dev.queueBuffer(buf);
    }
}

void prerollAudio(ALSADevice& dev, int audioBufSize, int buffers)
{
    int32_t* audio_buf = new int32_t[audioBufSize * AUDIO_CHANNELS];
    memset(audio_buf, 0, audioBufSize * AUDIO_CHANNELS * sizeof(int32_t));

    for (int i = 0; i < buffers; i++)
        snd_pcm_writei(dev.pcm().get(), audio_buf, audioBufSize);

    delete[] audio_buf;
}

void setupAudio(ALSADevice& dev)
{
    snd_pcm_hw_params_t* params = nullptr;
    snd_pcm_sw_params_t* sw_params = nullptr;
    snd_pcm_format_t format;
    snd_pcm_uframes_t period;
    unsigned int channels, rate;
    int dir;

    // Alloc hardware parameter structs
    snd_pcm_hw_params_malloc(&params);

    // Get default hardware parameters
    snd_pcm_hw_params_any(dev.pcm().get(), params);

    // Interleave channels
    snd_pcm_hw_params_set_access(dev.pcm().get(), params, SND_PCM_ACCESS_RW_INTERLEAVED);

    // Set period and buffer sizes
    snd_pcm_hw_params_set_period_time(dev.pcm().get(), params, AUDIO_PERIOD_SIZE_US, 0);
    snd_pcm_hw_params_set_buffer_time(dev.pcm().get(), params, AUDIO_BUFFER_SIZE_US, 0);

    // Set number of channels
    snd_pcm_hw_params_set_channels(dev.pcm().get(), params, AUDIO_CHANNELS);

    // Set hardware parameters, you can also change format, channels and other params prior this
    // call
    snd_pcm_hw_params(dev.pcm().get(), params);

    // Extract accepted parameters,
    snd_pcm_hw_params_get_format(params, &format);
    snd_pcm_hw_params_get_channels(params, &channels);
    snd_pcm_hw_params_get_rate(params, &rate, &dir);
    snd_pcm_hw_params_get_period_size(params, &period, &dir);

    snd_pcm_hw_params_free(params);

    // Alloc software parameter structs
    snd_pcm_sw_params_alloca(&sw_params);
    // Get current software parameters
    snd_pcm_sw_params_current(dev.pcm().get(), sw_params);
    // Enable ALSA timestamping so we can always sync audio with video
    snd_pcm_sw_params_set_tstamp_mode(dev.pcm().get(), sw_params, SND_PCM_TSTAMP_ENABLE);
    // Disable ALSA auto-start
    snd_pcm_sw_params_set_start_threshold(dev.pcm().get(), sw_params, LONG_MAX);
    // Set software parameters
    snd_pcm_sw_params(dev.pcm().get(), sw_params);

    snd_pcm_prepare(dev.pcm().get());

    std::cout << "ALSA playback " << channels << " x " << snd_pcm_format_name(format) << " x "
              << rate << " with period of " << period << " samples" << std::endl;
}

void mainLoop(V4L2Device& videoDev, ALSADevice& audioDev, const v4l2_pix_format_mplane& videoFormat)
{
    int videoType = videoDev.type();

    bool bottomField = false;
    int num = 0;

    // Start audio capture
    snd_pcm_start(audioDev.pcm().get());
    // Start capture, VBI device first!
    videoDev.ioctl(VIDIOC_STREAMON, &videoType);

    pollfd pollfd[1];

    pollfd[0].fd = videoDev.fd();
    pollfd[0].events = POLLOUT;

    while (!shouldStop) {
        // Wait for frame
        poll(pollfd, 1, 100);

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);
        uint64_t curTs = (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec;

        if (pollfd[0].revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Video/VBI frame sended
        if (pollfd[0].revents & POLLOUT) {
            // Get buffer from driver, video first.
            auto videoBuf = videoDev.dequeueBuffer();

            uint64_t vidTs = (uint64_t)videoBuf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + videoBuf->v4l2buf().timestamp.tv_usec * 1000;

            std::cout << "Sended video buffer at " << curTs << "ns, "
                      << "video id = " << videoBuf->v4l2buf().index
                      << ", video sequence = " << videoBuf->v4l2buf().sequence
                      << ", video field = " << videoBuf->v4l2buf().field
                      << ", video timestamp = " << vidTs
                      << "ns, video size = " << videoBuf->bytesused(0) << std::endl;

            uint8_t* videoData = videoBuf->data(0);
            int32_t* audioData = nullptr;

            if (videoFormat.field != V4L2_FIELD_INTERLACED) {
                videoBuf->v4l2buf().field = (videoFormat.field == V4L2_FIELD_NONE)
                    ? V4L2_FIELD_NONE
                    : (bottomField ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP);
                bottomField = !bottomField;

                int n = num % (TEST_PERIOD_FIELDS);
                uint8_t nY = (uint8_t)(16 + n * 235 / TEST_PERIOD_FIELDS);
                for (int i = 0; i < videoBuf->length() / 2; i++) {
                    videoData[i * 2 + 0] = 0x80;
                    videoData[i * 2 + 1] = nY;
                }
                audioData = ((num % TEST_PERIOD_FIELDS) == 0) ? audioBufSin : audioBufSilence;
            } else {
                int n = num % (TEST_PERIOD_FIELDS / 2) * 2;
                uint8_t nYO = (uint8_t)(16 + (n + 0) * 235 / TEST_PERIOD_FIELDS);
                uint8_t nYE = (uint8_t)(16 + (n + 1) * 235 / TEST_PERIOD_FIELDS);

                for (int l = 0; l < videoFormat.height; l++) {
                    for (int x = 0; x < videoFormat.width; x++) {
                        videoData[l * videoFormat.plane_fmt[0].bytesperline + x * 2 + 0] = 0x80;
                        videoData[l * videoFormat.plane_fmt[0].bytesperline + x * 2 + 1]
                            = (l % 2) ? nYE : nYO;
                    }
                }
                audioData = ((num % (TEST_PERIOD_FIELDS / 2)) == 0) ? audioBufSin : audioBufSilence;
            }

            snd_pcm_sframes_t size = snd_pcm_writei(audioDev.pcm().get(), audioData, audioBufSize);

            std::cout << "Sending " << size << " audio frames at " << curTs << "ns, " << std::endl;

            videoDev.queueBuffer(videoBuf);

            num++;
        }
    }

    // Stop audio capture
    snd_pcm_drop(audioDev.pcm().get());
    // Stop capture, Video device first!
    videoDev.ioctl(VIDIOC_STREAMOFF, &videoType);
}

int main(int argc, char** argv)
{
    std::string videoPath, audioPath;
    if (argc != 3) {
        std::cout << "Usage: " << argv[0] << " video_device audio_device" << std::endl;
        return 0;
    }

    videoPath = argv[1];
    std::cout << "Openning " << videoPath << std::endl;
    V4L2Device videoDev(videoPath);

    if (!videoDev.isOpen()) {
        std::cerr << "Cannot open video device: " << videoDev.errorString() << std::endl;
        return -1;
    }

    if (videoDev.type() != V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE) {
        std::cerr << "Device " << videoPath << " is not an output video device" << std::endl;
        return -1;
    }

    audioPath = argv[2];
    ALSADevice audioDev(audioPath, SND_PCM_STREAM_PLAYBACK);

    if (!audioDev.isOpen()) {
        std::cerr << "Cannot open audio device: " << audioDev.errorString() << std::endl;
        return -1;
    }

    setupAudio(audioDev);

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format videoFormat;

    videoFormat = setupVideoFormat(videoDev);
    videoDev.requestBuffers(BUFFER_COUNT);

    // Alloc audio buffer
    audioBufSize = getAudioPerVideoFrame(videoDev, videoFormat.fmt.pix_mp);
    audioBufSilence = new int32_t[audioBufSize * AUDIO_CHANNELS];
    audioBufSin = new int32_t[audioBufSize * AUDIO_CHANNELS];

    memset(audioBufSin, 0, audioBufSize * AUDIO_CHANNELS * 4);
    memset(audioBufSilence, 0, audioBufSize * AUDIO_CHANNELS * 4);

    for (snd_pcm_uframes_t i = 0; i < 96; i++) {
        float v = (float)sin(M_PI * (i + 1) / 48);
        audioBufSin[i * 2 + 0] = v * (float)((1U << 31) - 1);
        audioBufSin[i * 2 + 1] = v * (float)((1U << 31) - 1);
    }

    std::cout << "Playback with " << videoDev.buffers().size() << " buffers" << std::endl;

    // Preroll video
    preroll(videoDev, videoDev.buffers(), videoFormat.fmt.pix_mp);
    prerollAudio(audioDev, getAudioPerVideoFrame(videoDev, videoFormat.fmt.pix_mp),
        videoDev.buffers().size());

    mainLoop(videoDev, audioDev, videoFormat.fmt.pix_mp);

    delete[] audioBufSilence;
    delete[] audioBufSin;

    videoDev.freeBuffers();

    return 0;
}
