#include "ALSADevice.hpp"
#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-alsa-ioctl.h"
#include "forward-v4l2-ioctl.h"

#include <arpa/inet.h>
#include <climits>
#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <list>
#include <net/if.h>
#include <poll.h>
#include <regex>
#include <sys/ioctl.h>
#include <tuple>

/*!
 * \dir ./
 * This example shows video + audio capture example on FD2110
 */

namespace {
static const int BUFFER_COUNT = 8;
// ALSA chunk size, should be less than frame (field) period
static const int AUDIO_PERIOD_SIZE_US = 10000;
static const int AUDIO_BUFFER_SIZE_US = 1000000;
static const int AUDIO_CHANNELS = 2;
static const snd_pcm_uframes_t AUDIO_BUFFER_SIZE = 2048 * 16;
}

static bool shouldStop = false;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

bool str2ipv6(const std::string& str, uint8_t* ip, bool& v6)
{
    struct in_addr addr;
    struct in6_addr addr6;

    memset(ip, 0, 16);

    if (inet_pton(AF_INET, str.c_str(), &addr)) {
        uint8_t* in_p = (uint8_t*)&addr.s_addr;

        for (int i = 0; i < 4; i++)
            ip[12 + i] = in_p[i];
        ip[11] = 0xFF;
        ip[10] = 0xFF;
        v6 = false;
    } else if (inet_pton(AF_INET6, str.c_str(), &addr6)) {
        for (int i = 0; i < 16; i++)
            ip[i] = addr6.s6_addr[i];
        v6 = true;
    } else
        return false;

    return true;
}

int setupVideoStream(V4L2Device& dev, const std::string& videoAddr, uint16_t videoPort)
{
    struct v4l2_forward_stream_address v4l2addr;
    bool v6;

    if (!str2ipv6(videoAddr, v4l2addr.ipv6, v6)) {
        std::cerr << "Cannot parse video IP" << std::endl;
        return -EINVAL;
    }
    v4l2addr.port = videoPort;

    struct v4l2_ext_control ex_ctl;
    struct v4l2_ext_controls ex_ctls;
    struct v4l2_control ctl;

    // Setup IP + Port
    memset(&ex_ctl, 0, sizeof(ex_ctl));
    memset(&ex_ctls, 0, sizeof(ex_ctls));
    memset(&ctl, 0, sizeof(ctl));

    ex_ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_ADDRESS;
    ex_ctl.size = sizeof(v4l2addr);
    ex_ctl.ptr = &v4l2addr;
    ex_ctls.ctrl_class = V4L2_CTRL_ID2CLASS(V4L2_CID_FORWARD_FD2110_STREAM_ADDRESS);
    ex_ctls.count = 1;
    ex_ctls.controls = &ex_ctl;

    if (dev.ioctl(VIDIOC_S_EXT_CTRLS, &ex_ctls) < 0) {
        std::cerr << "Cannot set video stream address: " << dev.errorString() << std::endl;
        return dev.error();
    }

    // Setup input stream color mode + bit depth - should be same as in SDP.
    // Board can do some conversion between stream and v4l2 device format (e.g. 10->8 bit
    //   conversion).
    // Set YCbCr 4:2:2, 10-bit - default format for SDI->2110 converters like AJA.
    ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_COLOR;
    ctl.value = V4L_FORWARD_FD2110_STREAM_COLOR_YCBCR422;
    if (dev.ioctl(VIDIOC_S_CTRL, &ctl) < 0) {
        std::cerr << "Cannot set video stream color format: " << dev.errorString() << std::endl;
        return dev.error();
    }

    ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_BPC;
    ctl.value = V4L_FORWARD_FD2110_STREAM_BPC_10BIT;
    if (dev.ioctl(VIDIOC_S_CTRL, &ctl) < 0) {
        std::cerr << "Cannot set video stream bpc: " << dev.errorString() << std::endl;
        return dev.error();
    }

    return 0;
}

int alsaControlLookup(
    snd_ctl_t* dev, snd_ctl_elem_id_t* id, const std::string& name, int io, int stream)
{
    snd_ctl_elem_info_t* info;
    snd_ctl_elem_info_alloca(&info);

    snd_ctl_elem_id_set_interface(id, SND_CTL_ELEM_IFACE_PCM);
    snd_ctl_elem_id_set_device(id, io);
    snd_ctl_elem_id_set_subdevice(id, stream);
    snd_ctl_elem_id_set_name(id, name.c_str());

    snd_ctl_elem_info_set_id(info, id);
    int result = snd_ctl_elem_info(dev, info);
    if (result) {
        std::cerr << "Can't find ALSA control \"" << name << "\": " << snd_strerror(result)
                  << std::endl;
        return result;
    }
    snd_ctl_elem_info_get_id(info, id);

    return 0;
}

int alsaControlWriteArray(
    snd_ctl_t* dev, int io, int stream, const std::string& name, const uint8_t* data, size_t size)
{
    snd_ctl_elem_value_t* value;
    snd_ctl_elem_value_alloca(&value);
    snd_ctl_elem_id_t* id;
    snd_ctl_elem_id_alloca(&id);
    int result;

    result = alsaControlLookup(dev, id, name, io, stream);
    if (result)
        return result;

    snd_ctl_elem_value_set_id(value, id);
    for (size_t i = 0; i < size; i++)
        snd_ctl_elem_value_set_byte(value, i, data[i]);
    result = snd_ctl_elem_write(dev, value);
    if (result) {
        std::cerr << "Cannot write ALSA control: " << snd_strerror(result) << std::endl;
        return result;
    }
    return result;
}

int alsaControlWriteEnum(snd_ctl_t* dev, int io, int stream, const std::string& name, int v)
{
    snd_ctl_elem_value_t* value;
    snd_ctl_elem_value_alloca(&value);
    snd_ctl_elem_id_t* id;
    snd_ctl_elem_id_alloca(&id);
    int result;

    result = alsaControlLookup(dev, id, name, io, stream);
    if (result)
        return result;

    snd_ctl_elem_value_set_id(value, id);
    snd_ctl_elem_value_set_enumerated(value, 0, v);
    result = snd_ctl_elem_write(dev, value);
    if (result) {
        std::cerr << "Cannot write ALSA control: " << snd_strerror(result) << std::endl;
        return result;
    }
    return result;
}

int alsaControlWriteInt(snd_ctl_t* dev, int io, int stream, const std::string& name, int v)
{
    snd_ctl_elem_value_t* value;
    snd_ctl_elem_value_alloca(&value);
    snd_ctl_elem_id_t* id;
    snd_ctl_elem_id_alloca(&id);
    int result;

    result = alsaControlLookup(dev, id, name, io, stream);
    if (result)
        return result;

    snd_ctl_elem_value_set_id(value, id);
    snd_ctl_elem_value_set_integer(value, 0, v);
    result = snd_ctl_elem_write(dev, value);
    if (result) {
        std::cerr << "Cannot write ALSA control: " << snd_strerror(result) << std::endl;
        return result;
    }
    return result;
}

int setupAudioStream(
    ALSADevice& dev, const std::string& audioAddr, uint16_t audioPort, uint8_t audioPT)
{
    int result;
    struct snd_forward_stream_address addr;
    bool v6;

    if (!str2ipv6(audioAddr, addr.ipv6, v6)) {
        std::cerr << "Cannot parse audio IP" << std::endl;
        return -EINVAL;
    }
    addr.port = audioPort;
    addr.payload_type = audioPT;

    // Get sound card and io number from PCM device
    snd_pcm_info_t* pcm_info;
    snd_pcm_info_alloca(&pcm_info);
    result = snd_pcm_info(dev.pcm().get(), pcm_info);
    if (result) {
        std::cerr << "Cannot get ALSA PCM info: " << snd_strerror(result) << std::endl;
        return result;
    }

    int card = snd_pcm_info_get_card(pcm_info);
    int io = snd_pcm_info_get_device(pcm_info);

    // Open control device
    snd_ctl_t* ctl;
    std::string ctl_name = std::string("hw:") + std::to_string(card);
    snd_ctl_open(&ctl, ctl_name.c_str(), 0);

    // Set stream address control for device
    result = alsaControlWriteArray(ctl, io, SND_PCM_STREAM_CAPTURE, "PCM Capture Stream Address",
        (uint8_t*)&addr, sizeof(addr));
    if (result)
        return result;

    // Set stream format - L24
    result = alsaControlWriteEnum(ctl, io, SND_PCM_STREAM_CAPTURE, "PCM Capture Stream Format",
        SND_FORWARD_STREAM_FORMAT_L24);
    if (result)
        return result;

    // Set stream number of channels - 4 = one SDI group
    result = alsaControlWriteInt(ctl, io, SND_PCM_STREAM_CAPTURE, "PCM Capture Stream Channels", 4);
    if (result)
        return result;

    snd_ctl_close(ctl);

    return 0;
}

int openSocket(const std::string& addr, uint16_t port, const std::string& iface)
{
    int sockfd;
    int optval;
    struct sockaddr_in6 saddr;
    int result;
    bool v6;

    // Set socket address
    memset(&saddr, 0, sizeof(saddr));
    if (!str2ipv6(addr, saddr.sin6_addr.s6_addr, v6)) {
        std::cerr << "Cannot parse IP" << std::endl;
        return -EINVAL;
    }
    saddr.sin6_family = AF_INET6;
    saddr.sin6_port = htons(port);

    // Open socket
    sockfd = socket(AF_INET6, SOCK_DGRAM, 0);
    if (sockfd < 0) {
        std::cerr << "Cannot open socket: " << strerror(errno) << std::endl;
        return -errno;
    }

    // Allow both IPv6 and IPv4 on same socket
    optval = 0;
    result = setsockopt(sockfd, IPPROTO_IPV6, IPV6_V6ONLY, &optval, sizeof(optval));
    if (result) {
        std::cerr << "IPV6_V6ONLY failed: " << strerror(errno) << std::endl;
        return result;
    }

    // Bind socket to device
    result = setsockopt(sockfd, SOL_SOCKET, SO_BINDTODEVICE, iface.c_str(), iface.length());
    if (result) {
        std::cerr << "SO_BINDTODEVICE failed: " << strerror(errno) << std::endl;
        return result;
    }

    // Bind socket to address
    result = bind(sockfd, (sockaddr*)&saddr, sizeof(saddr));
    if (result) {
        std::cerr << "Socket binding failed: " << strerror(errno) << std::endl;
        return result;
    }

    // Find net interface index by name
    struct ifreq ifidx;
    memset(&ifidx, 0, sizeof(ifidx));
    strncpy(ifidx.ifr_name, iface.c_str(), IFNAMSIZ - 1);
    result = ioctl(sockfd, SIOCGIFINDEX, &ifidx);
    if (result < 0) {
        std::cerr << "Cannot find net interface index" << std::endl;
        return -EINVAL;
    }

    // Allow address reuse
    optval = 1;
    result = setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval));
    if (result) {
        std::cerr << "SO_REUSEADDR failed: " << strerror(errno) << std::endl;
        return result;
    }

    // Join multicast
    if (v6) {
        struct ipv6_mreq mreq;
        mreq.ipv6mr_multiaddr = saddr.sin6_addr;
        mreq.ipv6mr_interface = ifidx.ifr_ifindex;
        result = setsockopt(sockfd, IPPROTO_IPV6, IPV6_ADD_MEMBERSHIP, &mreq, sizeof(mreq));
        if (result) {
            std::cerr << "IPV6_ADD_MEMBERSHIP failed: " << strerror(errno) << std::endl;
            return result;
        }
    } else {
        struct ip_mreqn mreq;
        mreq.imr_multiaddr.s_addr = *((in_addr_t*)&saddr.sin6_addr.s6_addr[12]);
        mreq.imr_address.s_addr = INADDR_ANY;
        mreq.imr_ifindex = ifidx.ifr_ifindex;
        result = setsockopt(sockfd, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq));
        if (result) {
            std::cerr << "IP_ADD_MEMBERSHIP failed: " << strerror(errno) << std::endl;
            return result;
        }
    }

    return sockfd;
}

int setupStream(const std::string& board, int stream, const std::string& videoAddr,
    uint16_t videoPort, const std::string& audioAddr, uint16_t audioPort, uint8_t audioPT,
    std::string& videoDev, std::string& audioDev, int& socketfd)
{
    Board::Info info = Board::fromPath(board);
    int result = 0;

    if (info.type == Board::Info::Unknown) {
        std::cerr << "Wrong board path, should be /dev/forward/fd2110-*" << std::endl;
        return -ENODEV;
    } else if (info.type != Board::Info::FD2110) {
        std::cerr << "Not a FD2110 board" << std::endl;
        return -EINVAL;
    }

    Board fd2110(info);

    // Enable board stream
    std::cout << "Enabling input stream #" << stream << "..." << std::endl;
    result = fd2110.switchIOMode(stream + 2, Board::IOMode::Input);
    if (result) {
        std::cerr << "Cannot enable stream #" << stream << ": " << strerror(result) << std::endl;
        return result;
    }

    // Getting devices
    info = fd2110.info();

    videoDev = info.videoDevs[stream + 2];
    audioDev = info.audioDevs[stream + 2];

    std::cout << "Stream #" << stream << " enabled, video device: " << videoDev
              << ", audio device: " << audioDev << std::endl;

    // Setup V4L2 device - pass stream address and format to driver
    V4L2Device video(videoDev);
    if (!video.isOpen()) {
        std::cerr << "Cannot open video device: " << video.errorString() << std::endl;
        return -1;
    }
    std::cout << "Configuring V4L2 device..." << std::endl;
    result = setupVideoStream(video, videoAddr, videoPort);
    if (result)
        return result;

    // Setup ALSA device - pass stream address, payload type and format to driver
    ALSADevice audio(audioDev, SND_PCM_STREAM_CAPTURE);
    if (!audio.isOpen()) {
        std::cerr << "Cannot open audio device: " << audio.errorString() << std::endl;
        return -1;
    }
    std::cout << "Configuring ALSA device..." << std::endl;
    result = setupAudioStream(audio, audioAddr, audioPort, audioPT);
    if (result)
        return result;

    // Setup network - open socket for IP filter setup and multicast join
    socketfd = openSocket(videoAddr, videoPort, info.netDevs[stream + 2]);
    if (socketfd < 0)
        return -socketfd;

    return 0;
}

v4l2_format setupVideoFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;
    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;
    // 8-bit YUYV
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_YUYV;
    // 10-bit v210
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_V210;
    // Raw, 10-bit dense packed
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Capture frame format is " << fmt << std::endl;

    return fmt;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers)
{
    // Fill capture queue
    for (auto buf : buffers) {
        dev.queueBuffer(buf);
    }
}

void setupAudio(ALSADevice& dev)
{
    snd_pcm_hw_params_t* params = nullptr;
    snd_pcm_sw_params_t* sw_params = nullptr;
    int err = 0;
    snd_pcm_format_t format;
    snd_pcm_uframes_t period;
    unsigned int channels, rate;
    int dir = 0;

    // Alloc hardware parameter structs
    snd_pcm_hw_params_malloc(&params);

    // Get default hardware parameters
    snd_pcm_hw_params_any(dev.pcm().get(), params);

    // Interleave channels
    snd_pcm_hw_params_set_access(dev.pcm().get(), params, SND_PCM_ACCESS_RW_INTERLEAVED);

    // Set number of channels
    snd_pcm_hw_params_set_channels(dev.pcm().get(), params, AUDIO_CHANNELS);

    // Set period and buffer sizes
    snd_pcm_hw_params_set_period_time(dev.pcm().get(), params, AUDIO_PERIOD_SIZE_US, 0);
    snd_pcm_hw_params_set_buffer_time(dev.pcm().get(), params, AUDIO_BUFFER_SIZE_US, 0);

    // Set hardware parameters, you can also change format, channels and other params prior this
    // call
    snd_pcm_hw_params(dev.pcm().get(), params);

    snd_pcm_hw_params_free(params);

    // Extract accepted parameters,
    snd_pcm_hw_params_get_format(params, &format);
    snd_pcm_hw_params_get_channels(params, &channels);
    snd_pcm_hw_params_get_rate(params, &rate, &dir);
    snd_pcm_hw_params_get_period_size(params, &period, &dir);

    // Alloc software parameter structs
    snd_pcm_sw_params_alloca(&sw_params);
    // Get current software parameters
    snd_pcm_sw_params_current(dev.pcm().get(), sw_params);
    // Enable ALSA timestamping so we can always sync audio with video
    snd_pcm_sw_params_set_tstamp_mode(dev.pcm().get(), sw_params, SND_PCM_TSTAMP_ENABLE);
    // Disable ALSA auto-start
    snd_pcm_sw_params_set_start_threshold(dev.pcm().get(), sw_params, LONG_MAX);
    // Set software parameters
    snd_pcm_sw_params(dev.pcm().get(), sw_params);

    snd_pcm_prepare(dev.pcm().get());

    std::cout << "ALSA capture " << channels << " x " << snd_pcm_format_name(format) << " x "
              << rate << " with period of " << period << " samples" << std::endl;
}

// Common capture code - same as in capture-vbi-audio sample
void mainLoop(V4L2Device& videoDev, ALSADevice& audioDev, const v4l2_pix_format_mplane& videoFormat)
{
    int videoType = videoDev.type();

    // Alloc and setup ALSA status structure - used to get current ALSA status
    std::shared_ptr<snd_pcm_status_t> status;
    snd_pcm_status_t* sp;
    snd_pcm_status_malloc(&sp);
    status = std::shared_ptr<snd_pcm_status_t>(sp, snd_pcm_status_free);

    // Forward-specific setup - this will tell ALSA and driver to return real timestamp in status
    // This timestamp will be same as in corresponding video buffer
    snd_pcm_audio_tstamp_config_t config;
    config.report_delay = 0;
    config.type_requested = SND_PCM_AUDIO_TSTAMP_TYPE_LINK_ABSOLUTE;
    snd_pcm_status_set_audio_htstamp_config(status.get(), &config);

    // Alloc audio buffer
    int32_t* audio_buf = new int32_t[AUDIO_BUFFER_SIZE * AUDIO_CHANNELS];

    // Start audio capture
    snd_pcm_start(audioDev.pcm().get());
    // Start video capture
    int result = videoDev.ioctl(VIDIOC_STREAMON, &videoType);

    // Tricky polling setup, so we can use single poll() both for audio and video
    int alsaPollCount = snd_pcm_poll_descriptors_count(audioDev.pcm().get());

    pollfd pollfd[alsaPollCount + 1];

    pollfd[0].fd = videoDev.fd();
    pollfd[0].events = POLLIN;

    snd_pcm_poll_descriptors(audioDev.pcm().get(), &pollfd[1], alsaPollCount);

    // Small queue (buffer) to sync video with vbi and audio
    // In normal situation poll() will return POLLIN both for video and audio at same time,
    //   but under heavy load it can return either video or audio first
    // So we can use small buffer to sync them back
    std::list<std::tuple<uint64_t, V4L2Device::BufferPtr, std::vector<int32_t>>> queue;

    while (!shouldStop) {
        // Wait for frame
        poll(pollfd, alsaPollCount + 1, 100);

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);
        uint64_t curTs = (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec;

        if (pollfd[0].revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Video/VBI frame arrived
        if (pollfd[0].revents & POLLIN) {
            // Get buffer from driver
            auto videoBuf = videoDev.dequeueBuffer();

            uint64_t vidTs = (uint64_t)videoBuf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + videoBuf->v4l2buf().timestamp.tv_usec * 1000;

            std::cout << "Received video buffer at " << curTs << "ns, "
                      << "video id = " << videoBuf->v4l2buf().index
                      << ", video sequence = " << videoBuf->v4l2buf().sequence
                      << ", video field = " << videoBuf->v4l2buf().field
                      << ", video timestamp = " << vidTs
                      << "ns, video size = " << videoBuf->v4l2buf().bytesused << std::endl;

            // Find queue entry with same timestamp as video
            auto qit = queue.end();
            for (auto it = queue.begin(); it != queue.end(); it++) {
                if (std::get<0>(*it) == vidTs) {
                    qit = it;
                    break;
                }
            }

            // If no entry with same timestamp, add it, update video buffer otherwise
            if (qit == queue.end())
                queue.push_back(std::make_tuple(vidTs, videoBuf, std::vector<int32_t>()));
            else
                std::get<1>(*qit) = videoBuf;
        }

        // Usually, there is only one pollfd for ALSA
        if (pollfd[1].revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Audio chunk arrived
        if (pollfd[1].revents & POLLIN) {
            // Get current ALSA status
            snd_pcm_status(audioDev.pcm().get(), status.get());
            // .. and timestamp
            snd_htimestamp_t ts;
            snd_pcm_status_get_audio_htstamp(status.get(), &ts);

            // Timestamp, drop ns part as in V4L2
            uint64_t audioTs = (uint64_t)ts.tv_sec * 1000000000ULL + (ts.tv_nsec / 1000) * 1000;

            // Number of frames available
            snd_pcm_uframes_t avail = snd_pcm_status_get_avail(status.get());
            snd_pcm_sframes_t size = snd_pcm_readi(
                audioDev.pcm().get(), audio_buf, std::min(avail, AUDIO_BUFFER_SIZE));

            std::cout << "Received " << size << " audio frames at " << curTs << "ns, "
                      << "audio timestamp = " << audioTs << std::endl;

            // Find place in queue - same as video
            auto qit = queue.end();
            for (auto it = queue.begin(); it != queue.end(); it++) {
                if (std::get<0>(*it) == audioTs) {
                    qit = it;
                    break;
                }
            }
            std::vector<int32_t> data(audio_buf, audio_buf + size * AUDIO_CHANNELS);
            if (qit == queue.end())
                queue.push_back(std::make_tuple(audioTs, nullptr, data));
            else
                std::get<2>(*qit) = data;
        }

        // Now we have queue with Video and Audio, work with them
        while (!queue.empty()) {
            auto& entry = queue.front();
            bool drop = false;
            bool haveData = false;

            // Data to process
            uint64_t timestamp = std::get<0>(entry);
            V4L2Device::BufferPtr video = std::get<1>(entry);
            std::vector<int32_t>& audio = std::get<2>(entry);

            haveData = (video && !audio.empty());

            // If there more than one entry in queue - we have problems, either v4l2 or audio
            // buffers is dropped
            if ((int)queue.size() > 1) {
                std::cout << "Video/Audio desync at ts = " << timestamp << "ns";

                // !.type means buffer is invalid
                if (!video)
                    std::cout << ", no video";
                if (audio.empty())
                    std::cout << ", no audio";
                std::cout << std::endl;

                drop = true;
            }

            if (drop || haveData) {
                /*
                 * PROCESS DATA HERE
                 */
                std::cout << "Got synced audio, video with timestamp = " << timestamp << "ns"
                          << std::endl;

                // Return buffers back to queue
                videoDev.queueBuffer(video);

                queue.pop_front();
            } else if (!drop && !haveData) {
                // Not ready yet
                break;
            }
        }
    }

    // Stop audio capture
    snd_pcm_drop(audioDev.pcm().get());
    // Stop capture, Video device first!
    videoDev.ioctl(VIDIOC_STREAMOFF, &videoType);
}

int main(int argc, char** argv)
{
    std::string board, videoAddr, audioAddr;
    int stream;
    int ethernet;
    uint8_t audioPT;
    uint16_t videoPort, audioPort;
    std::string videoPath, audioPath;
    int socketfd;

    if (argc != 8) {
        std::cout << "Usage: " << argv[0]
                  << " board stream_idx video_addr video_port audio_addr audio_port audio_pt"
                  << std::endl;
        return 0;
    }

    // FD2110 has (16 input streams + 16 output streams) x 2 Ethernets = 64 streams total
    // Streams represented as I/O pin, in order: 2x SDI, 16x eth0 inputs, 16x eth0 outputs, 16x eth1
    // inputs, 16x eth1 outputs
    board = argv[1];
    stream = std::stoi(argv[2]);
    videoAddr = argv[3];
    videoPort = std::stoul(argv[4]);
    audioAddr = argv[5];
    audioPort = std::stoul(argv[6]);
    audioPT = std::stoi(argv[7]);

    // Setup stream parameters - addresses, ports, payload type, pixel format, audio format, etc.
    int result = setupStream(board, stream, videoAddr, videoPort, audioAddr, audioPort, audioPT,
        videoPath, audioPath, socketfd);

    // Common capture code below...
    std::cout << "Openning " << videoPath << std::endl;
    V4L2Device videoDev(videoPath);

    if (!videoDev.isOpen()) {
        std::cerr << "Cannot open video device: " << videoDev.errorString() << std::endl;
        return -1;
    }

    if (videoDev.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device " << videoPath << " is not an input video device" << std::endl;
        return -1;
    }

    ALSADevice audioDev(audioPath, SND_PCM_STREAM_CAPTURE);

    if (!audioDev.isOpen()) {
        std::cerr << "Cannot open audio device: " << audioDev.errorString() << std::endl;
        return -1;
    }

    setupAudio(audioDev);

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    std::vector<V4L2Device::Buffer> videoBuffers;
    v4l2_format videoFormat;

    videoFormat = setupVideoFormat(videoDev);

    videoDev.requestBuffers(BUFFER_COUNT);

    std::cout << "Capturing with " << videoDev.buffers().size() << " buffers" << std::endl;

    preroll(videoDev, videoDev.buffers());
    mainLoop(videoDev, audioDev, videoFormat.fmt.pix_mp);

    videoDev.freeBuffers();

    // Closing socket to leave multicast group
    if (socketfd > 0)
        close(socketfd);

    return 0;
}
