#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <csignal>
#include <cstring>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <poll.h>

/*!
 * \dir ./
 * This example shows simple raw SDI capture example
 */

namespace {
static const int BUFFER_COUNT = 8;
static const int SAVE_FIELD_COUNT = 10;
}

static bool shouldStop = false;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

v4l2_format setupFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Each field are in separate buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;

    // Raw - whole SDI frame, with HANC, VANC, 10-bit dense packed
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    dev.setFormat(fmt);

    std::cout << "Capture frame format is " << fmt << std::endl;

    return fmt;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers)
{
    // Fill playback queue
    for (auto buf : buffers) {
        dev.queueBuffer(buf);
    }
}

void mainLoop(V4L2Device& dev, const v4l2_pix_format_mplane& fmt)
{
    int type = dev.type();

    std::vector<std::shared_ptr<uint8_t>> saveBuffers(SAVE_FIELD_COUNT);
    for (int i = 0; i < SAVE_FIELD_COUNT; i++)
        saveBuffers[i] = std::shared_ptr<uint8_t>(new uint8_t[fmt.plane_fmt[0].sizeimage]);

    // Start capture
    dev.ioctl(VIDIOC_STREAMON, &type);

    pollfd pollfd;
    pollfd.fd = dev.fd();
    pollfd.events = POLLIN;

    for (int field = -2; field < SAVE_FIELD_COUNT; field++) {
        // Wait for frame
        poll(&pollfd, 1, 100);

        if (pollfd.revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Timeout, no frame
        if (!(pollfd.revents & POLLIN))
            continue;

        // Get buffer from queue
        auto buf = dev.dequeueBuffer();

        /*
         * PROCESS BUFFER DATA HERE
         */

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);

        std::cout << "Received buffer at "
                  << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                  << "ns, id = " << buf->v4l2buf().index
                  << ", sequence = " << buf->v4l2buf().sequence
                  << ", field = " << buf->v4l2buf().field << ", timestamp = "
                  << (uint64_t)buf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + buf->v4l2buf().timestamp.tv_usec * 1000
                  << "ns, size = " << buf->bytesused(0) << std::endl;

        if (field >= 0)
            memcpy(saveBuffers[field % SAVE_FIELD_COUNT].get(), buf->data(0), buf->bytesused());

        // Return buffer back to queue
        dev.queueBuffer(buf);
    }

    for (int i = 0; i < SAVE_FIELD_COUNT; i++) {
        std::string filename
            = std::string("/tmp/raw_sdi_") + std::to_string(i) + std::string(".raw");
        std::cout << "Saving field " << i << " to " << filename << std::endl;
        std::ofstream f(filename);
        f.write((const char*)saveBuffers[i].get(), fmt.plane_fmt[0].sizeimage);
        f.close();
    }

    // Stop capture
    dev.ioctl(VIDIOC_STREAMOFF, &type);
}

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc != 2) {
        std::cout << "Usage: " << argv[0] << " device" << std::endl;
        return 0;
    }

    devPath = argv[1];
    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    if (dev.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device (" << dev.name() << ") is not an input device" << std::endl;
        return -1;
    }

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format format;

    format = setupFormat(dev);
    dev.requestBuffers(BUFFER_COUNT);

    std::cout << "Capturing with " << dev.buffers().size() << " buffers" << std::endl;

    preroll(dev, dev.buffers());
    mainLoop(dev, format.fmt.pix_mp);

    dev.freeBuffers();

    return 0;
}
