#include "ALSADevice.hpp"
#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <climits>
#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <list>
#include <poll.h>
#include <tuple>

/*!
 * \dir ./
 * This example shows video + vbi + audio capture example
 */

namespace {
static const int BUFFER_COUNT = 8;
// ALSA chunk size, should be less than frame (field) period
static const int AUDIO_PERIOD_SIZE_US = 10000;
static const int AUDIO_BUFFER_SIZE_US = 1000000;
static const int AUDIO_CHANNELS = 2;
static const snd_pcm_uframes_t AUDIO_BUFFER_SIZE = 2048 * 16;
}

static bool shouldStop = false;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

// VBI should be explicit enabled by controls, because it requires some overhead in driver
void enableVBI(V4L2Device& dev)
{
    struct v4l2_control ctl;
    ctl.id = V4L2_CID_FORWARD_ENABLE_VANC;
    ctl.value = true;

    if (dev.ioctl(VIDIOC_S_CTRL, &ctl))
        std::cerr << "Cannot enable VANC: " << dev.errorString() << std::endl;
}

v4l2_format setupVideoFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;
    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;
    // 8-bit YUYV
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_YUYV;
    // 10-bit v210
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_V210;
    // Raw - whole SDI frame, with HANC, VANC, 10-bit dense packed
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Capture frame format is " << fmt << std::endl;

    return fmt;
}

v4l2_format setupVBIFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // 10-bit data in 16-bit words, interleaved CYCYCYC as in SDI
    fmt.fmt.vbi.sample_format = V4L2_PIX_FMT_Y10;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Capture VBI format is " << fmt << std::endl;

    return fmt;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers)
{
    // Fill capture queue
    for (auto buf : buffers)
        dev.queueBuffer(buf);
}

void setupAudio(ALSADevice& dev)
{
    snd_pcm_hw_params_t* params = nullptr;
    snd_pcm_sw_params_t* sw_params = nullptr;
    int err = 0;
    snd_pcm_format_t format;
    snd_pcm_uframes_t period;
    unsigned int channels, rate;
    int dir = 0;

    // Alloc hardware parameter structs
    snd_pcm_hw_params_malloc(&params);

    // Get default hardware parameters
    snd_pcm_hw_params_any(dev.pcm().get(), params);

    // Interleave channels
    snd_pcm_hw_params_set_access(dev.pcm().get(), params, SND_PCM_ACCESS_RW_INTERLEAVED);

    // Set number of channels
    snd_pcm_hw_params_set_channels(dev.pcm().get(), params, AUDIO_CHANNELS);

    // Set period and buffer sizes
    snd_pcm_hw_params_set_period_time(dev.pcm().get(), params, AUDIO_PERIOD_SIZE_US, 0);
    snd_pcm_hw_params_set_buffer_time(dev.pcm().get(), params, AUDIO_BUFFER_SIZE_US, 0);

    // Set hardware parameters, you can also change format, channels and other params prior this
    // call
    snd_pcm_hw_params(dev.pcm().get(), params);

    snd_pcm_hw_params_free(params);

    // Extract accepted parameters,
    snd_pcm_hw_params_get_format(params, &format);
    snd_pcm_hw_params_get_channels(params, &channels);
    snd_pcm_hw_params_get_rate(params, &rate, &dir);
    snd_pcm_hw_params_get_period_size(params, &period, &dir);

    // Alloc software parameter structs
    snd_pcm_sw_params_alloca(&sw_params);
    // Get current software parameters
    snd_pcm_sw_params_current(dev.pcm().get(), sw_params);
    // Enable ALSA timestamping so we can always sync audio with video
    snd_pcm_sw_params_set_tstamp_mode(dev.pcm().get(), sw_params, SND_PCM_TSTAMP_ENABLE);
    // Disable ALSA auto-start
    snd_pcm_sw_params_set_start_threshold(dev.pcm().get(), sw_params, LONG_MAX);
    // Set software parameters
    snd_pcm_sw_params(dev.pcm().get(), sw_params);

    snd_pcm_prepare(dev.pcm().get());

    std::cout << "ALSA capture " << channels << " x " << snd_pcm_format_name(format) << " x "
              << rate << " with period of " << period << " samples" << std::endl;
}

void mainLoop(V4L2Device& videoDev, V4L2Device& vbiDev, ALSADevice& audioDev)
{
    int videoType = videoDev.type();
    int vbiType = vbiDev.type();

    // Alloc and setup ALSA status structure - used to get current ALSA status
    std::shared_ptr<snd_pcm_status_t> status;
    snd_pcm_status_t* sp;
    snd_pcm_status_malloc(&sp);
    status = std::shared_ptr<snd_pcm_status_t>(sp, snd_pcm_status_free);

    // Forward-specific setup - this will tell ALSA and driver to return real timestamp in status
    // This timestamp will be same as in corresponding video buffer
    snd_pcm_audio_tstamp_config_t config;
    config.report_delay = 0;
    config.type_requested = SND_PCM_AUDIO_TSTAMP_TYPE_LINK_ABSOLUTE;
    snd_pcm_status_set_audio_htstamp_config(status.get(), &config);

    // Alloc audio buffer
    int32_t* audio_buf = new int32_t[AUDIO_BUFFER_SIZE * AUDIO_CHANNELS];

    // Start audio capture
    snd_pcm_start(audioDev.pcm().get());
    // Start capture, VBI device first!
    vbiDev.ioctl(VIDIOC_STREAMON, &vbiType);
    videoDev.ioctl(VIDIOC_STREAMON, &videoType);

    // Tricky polling setup, so we can use single poll() both for audio and video
    int alsaPollCount = snd_pcm_poll_descriptors_count(audioDev.pcm().get());

    pollfd pollfd[alsaPollCount + 1];

    pollfd[0].fd = videoDev.fd();
    pollfd[0].events = POLLIN;

    snd_pcm_poll_descriptors(audioDev.pcm().get(), &pollfd[1], alsaPollCount);

    // Small queue (buffer) to sync video with vbi and audio
    // In normal situation poll() will return POLLIN both for video and audio at same time,
    //   but under heavy load it can return either video or audio first
    // So we can use small buffer to sync them back
    std::list<
        std::tuple<uint64_t, V4L2Device::BufferPtr, V4L2Device::BufferPtr, std::vector<int32_t>>>
        queue;

    while (!shouldStop) {
        // Wait for frame
        poll(pollfd, alsaPollCount + 1, 100);

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);
        uint64_t curTs = (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec;

        if (pollfd[0].revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Video/VBI frame arrived
        if (pollfd[0].revents & POLLIN) {
            // Get buffer from driver, video first.
            // Video and VBI is always in sync if number of queued buffers are always same
            // To be sure, you can check sequence fields
            V4L2Device::BufferPtr video = videoDev.dequeueBuffer();
            V4L2Device::BufferPtr vbi = vbiDev.dequeueBuffer();

            uint64_t vidTs = (uint64_t)video->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + video->v4l2buf().timestamp.tv_usec * 1000;
            uint64_t vbiTs = (uint64_t)vbi->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + vbi->v4l2buf().timestamp.tv_usec * 1000;

            std::cout << "Received video buffer at " << curTs << "ns, "
                      << "video id = " << video->v4l2buf().index
                      << ", video sequence = " << video->v4l2buf().sequence
                      << ", video field = " << video->v4l2buf().field
                      << ", video timestamp = " << vidTs
                      << "ns, video size = " << video->v4l2buf().m.planes[0].bytesused
                      << ", vbi id = " << vbi->v4l2buf().index
                      << ", vbi sequence = " << vbi->v4l2buf().sequence
                      << ", vbi field = " << vbi->v4l2buf().field << ", vbi timestamp = " << vbiTs
                      << "ns, vbi size = " << vbi->v4l2buf().bytesused << std::endl;

            // Find queue entry with same timestamp as video
            auto qit = queue.end();
            for (auto it = queue.begin(); it != queue.end(); it++) {
                if (std::get<0>(*it) == vidTs) {
                    qit = it;
                    break;
                }
            }

            // If no entry with same timestamp, add it, update video buffer otherwise
            if (qit == queue.end())
                queue.push_back(std::make_tuple(vidTs, video, nullptr, std::vector<int32_t>()));
            else
                std::get<1>(*qit) = video;

            // Same for VBI
            qit = queue.end();
            for (auto it = queue.begin(); it != queue.end(); it++) {
                if (std::get<0>(*it) == vbiTs) {
                    qit = it;
                    break;
                }
            }
            if (qit == queue.end())
                queue.push_back(std::make_tuple(vidTs, nullptr, vbi, std::vector<int32_t>()));
            else
                std::get<2>(*qit) = vbi;
        }

        // Usually, there is only one pollfd for ALSA
        if (pollfd[1].revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Audio chunk arrived
        if (pollfd[1].revents & POLLIN) {
            // Get current ALSA status
            snd_pcm_status(audioDev.pcm().get(), status.get());
            // .. and timestamp
            snd_htimestamp_t ts;
            snd_pcm_status_get_audio_htstamp(status.get(), &ts);

            // Timestamp, drop ns part as in V4L2
            uint64_t audioTs = (uint64_t)ts.tv_sec * 1000000000ULL + (ts.tv_nsec / 1000) * 1000;

            // Number of frames available
            snd_pcm_uframes_t avail = snd_pcm_status_get_avail(status.get());
            snd_pcm_sframes_t size = snd_pcm_readi(
                audioDev.pcm().get(), audio_buf, std::min(avail, AUDIO_BUFFER_SIZE));

            std::cout << "Received " << size << " audio frames at " << curTs << "ns, "
                      << "audio timestamp = " << audioTs << std::endl;

            // Find place in queue - same as video/VBI
            auto qit = queue.end();
            for (auto it = queue.begin(); it != queue.end(); it++) {
                if (std::get<0>(*it) == audioTs) {
                    qit = it;
                    break;
                }
            }
            std::vector<int32_t> data(audio_buf, audio_buf + size * AUDIO_CHANNELS);
            if (qit == queue.end())
                queue.push_back(std::make_tuple(audioTs, nullptr, nullptr, data));
            else
                std::get<3>(*qit) = data;
        }

        // Now we have queue with Video, VBI and Audio, work with them
        while (!queue.empty()) {
            auto& entry = queue.front();
            bool drop = false;
            bool haveData = false;

            // Data to process
            uint64_t timestamp = std::get<0>(entry);
            V4L2Device::BufferPtr video = std::get<1>(entry), vbi = std::get<2>(entry);
            std::vector<int32_t>& audio = std::get<3>(entry);

            haveData = (video && vbi && !audio.empty());

            // If there more than one entry in queue - we have problems, either v4l2 or audio
            // buffers is dropped
            if ((int)queue.size() > 1) {
                std::cout << "Video/VBI/Audio desync at ts = " << timestamp << "ns";

                // !.type means buffer is invalid
                if (!video)
                    std::cout << ", no video";
                if (!vbi)
                    std::cout << ", no VBI";
                if (audio.empty())
                    std::cout << ", no audio";
                std::cout << std::endl;

                drop = true;
            }

            if (drop || haveData) {
                /*
                 * PROCESS DATA HERE
                 */
                std::cout << "Got synced audio, video, vbi with timestamp = " << timestamp << "ns"
                          << std::endl;

                // Return buffers back to queue
                vbiDev.queueBuffer(vbi);
                videoDev.queueBuffer(video);

                queue.pop_front();
            } else if (!drop && !haveData) {
                // Not ready yet
                break;
            }
        }
    }

    // Stop audio capture
    snd_pcm_drop(audioDev.pcm().get());
    // Stop capture, Video device first!
    videoDev.ioctl(VIDIOC_STREAMOFF, &videoType);
    vbiDev.ioctl(VIDIOC_STREAMOFF, &vbiType);
}

int main(int argc, char** argv)
{
    std::string videoPath, vbiPath, audioPath;
    if (argc != 4) {
        std::cout << "Usage: " << argv[0] << " video_device vbi_device audio_device" << std::endl;
        return 0;
    }

    videoPath = argv[1];
    std::cout << "Openning " << videoPath << std::endl;
    V4L2Device videoDev(videoPath);

    if (!videoDev.isOpen()) {
        std::cerr << "Cannot open video device: " << videoDev.errorString() << std::endl;
        return -1;
    }

    if (videoDev.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device " << videoPath << " is not an input video device" << std::endl;
        return -1;
    }

    vbiPath = argv[2];
    std::cout << "Openning " << vbiPath << std::endl;
    V4L2Device vbiDev(vbiPath);

    if (!vbiDev.isOpen()) {
        std::cerr << "Cannot open VBI device: " << vbiDev.errorString() << std::endl;
        return -1;
    }

    if (vbiDev.type() != V4L2_BUF_TYPE_VBI_CAPTURE) {
        std::cerr << "Device " << vbiPath << " is not an input VBI device" << std::endl;
        return -1;
    }

    audioPath = argv[3];
    ALSADevice audioDev(audioPath, SND_PCM_STREAM_CAPTURE);

    if (!audioDev.isOpen()) {
        std::cerr << "Cannot open audio device: " << audioDev.errorString() << std::endl;
        return -1;
    }

    setupAudio(audioDev);

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format videoFormat, vbiFormat;

    enableVBI(videoDev);

    videoFormat = setupVideoFormat(videoDev);
    vbiFormat = setupVBIFormat(vbiDev);

    videoDev.requestBuffers(BUFFER_COUNT);
    vbiDev.requestBuffers(BUFFER_COUNT);

    std::cout << "Capturing with " << videoDev.buffers().size() << " buffers" << std::endl;

    preroll(vbiDev, vbiDev.buffers());
    preroll(videoDev, videoDev.buffers());
    mainLoop(videoDev, vbiDev, audioDev);

    videoDev.freeBuffers();
    vbiDev.freeBuffers();

    return 0;
}
