#include "ANCParser.hpp"

ANCParser::ANCParser()
{
}

void ANCParser::parse(uint8_t* data, int size, bool sd)
{
    for (auto& kv : m_data)
        kv.second.clear();
    for (auto& kv : m_packets)
        kv.second.clear();

    uint16_t* p = (uint16_t*)data;
    for (int i = 0; i < size / 2; i++) {
        uint8_t stream = (p[i] >> 10) & 0x7;
        uint16_t data = p[i] & 0x3FF;
        m_data[stream].push_back(data);
    }

    for (auto& kv : m_data) {
        uint8_t stream = kv.first;
        const std::vector<uint16_t>& data = kv.second;
        auto it = data.begin();
        int line = 0;

        while (std::distance(it, data.end()) >= 3) {
            if ((*(it + 0) == 0x3FF) && (*(it + 1) == 0x000) && (*(it + 2) == 0x000)) {
                if (std::distance(it, data.end()) < 6)
                    break;
                if (!sd) {
                    line = (*(it + 4) >> 2) & 0x7F;
                    line |= ((*(it + 5) >> 2) & 0x0F) << 7;
                    it += 6;
                } else
                    it += 4;
            } else if ((*(it + 0) == 0x000) && (*(it + 1) == 0x3FF) && (*(it + 2) == 0x3FF)) {
                if (std::distance(it, data.end()) < 7)
                    break;

                auto pit = it + 3;
                uint16_t csum = 0;

                ST291Header h;
                h.did = *pit & 0xFF;
                csum += *pit;
                pit++;
                h.sdid = *pit & 0xFF;
                csum += *pit;
                pit++;
                h.dc = *pit & 0xFF;
                csum += *pit;
                pit++;
                h.uwoffset = std::distance(data.begin(), pit);
                h.line = line;
                h.stream = stream;

                if (std::distance(pit, data.end()) < h.dc + 1)
                    break;

                for (int i = 0; i < h.dc; i++) {
                    csum += *pit;
                    pit++;
                }
                h.csok = (csum & 0x1FF) == (*pit & 0x1FF);

                if (h.csok) {
                    m_packets[stream].push_back(h);
                    it = ++pit;
                } else {
                    it++;
                }
            } else {
                it++;
            }
        }
    }
}

ANCParser::ST291Header ANCParser::findPacket(uint8_t stream, uint8_t did)
{
    int pos = 0;
    return findPacket(stream, did, pos);
}

ANCParser::ST291Header ANCParser::findPacket(uint8_t stream, uint8_t did, int& pos)
{
    ST291Header nullp { 0 };

    if (m_packets.find(stream) == m_packets.end()) {
        pos = -1;
        return nullp;
    }

    const std::vector<ST291Header>& packets = m_packets.at(stream);
    if (pos >= packets.size()) {
        pos = -1;
        return nullp;
    }

    for (; pos < packets.size(); pos++) {
        if ((packets[pos].did == did) && packets[pos].csok)
            return packets[pos];
    }
    pos = -1;
    return nullp;
}

int ANCParser::extractData8bit(const ST291Header& p, uint8_t* buf, int maxsize)
{
    if (p.dc == 0)
        return 0;

    int i;
    for (i = 0; (i < p.dc) && (i < maxsize); i++)
        buf[i] = m_data[p.stream][p.uwoffset + i] & 0xFF;
    return i;
}

int ANCParser::extractData8bit(uint8_t stream, uint8_t did, uint8_t* buf, int maxsize)
{
    int pos = 0;
    ST291Header h = findPacket(stream, did, pos);
    if (pos < 0)
        return 0;
    return extractData8bit(h, buf, maxsize);
}

int ANCParser::extractHDAudioPCM(uint32_t* samples, int channels, int maxsamples)
{
    // Extract ST299-1 PCM audio into 24 bit right-justified, removing all AES data & ECC
    int groups = (channels - 1) / 4 + 1;
    int pos[4] = { 0, 0, 0, 0 };
    uint8_t dbn[4] = { 0, 0, 0, 0 };
    for (const auto& h : m_packets[0]) {
        int group = (0xE7 - h.did);
        if (group < 0 || group >= groups)
            continue;

        if (pos[group] >= maxsamples)
            continue;

        uint8_t edbn = dbn[group] + 1;
        if (edbn == 0)
            edbn = 1;

        if (dbn[group] && h.sdid && (h.sdid != edbn)) {
            // !!! Samples drop !!!
        }

        for (int ch = 0; ch < 4; ch++) {
            uint16_t* p = &m_data[0].data()[h.uwoffset] + 2 + ch * 4;
            uint32_t s = 0;

            s |= uint32_t(*(p++) & 0xF0) >> 4;
            s |= uint32_t(*(p++) & 0xFF) << 4;
            s |= uint32_t(*(p++) & 0xFF) << 12;
            s |= uint32_t(*(p++) & 0x0F) << 20;

            if (group * 4 + ch < channels)
                samples[pos[group] * channels + ch] = s;
        }
        dbn[group] = h.sdid;
        pos[group]++;
    }
    // TODO: return all data counts
    return pos[0];
}

int ANCParser::extractSDAudioPCM(uint32_t* samples, int channels, int maxsamples)
{
    // Extract ST272 PCM audio into 24 bit right-justified, removing all AES data & ECC
    int groups = (channels - 1) / 4 + 1;
    int pos[16] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    uint8_t dbn[4] = { 0, 0, 0, 0 };
    for (const auto& h : m_packets[0]) {
        int group = (0xFF - h.did) / 2;
        if (group < 0 || group >= groups)
            continue;

        if (pos[group] >= maxsamples)
            continue;

        uint8_t edbn = dbn[group] + 1;
        if (edbn == 0)
            edbn = 1;

        if (dbn[group] && h.sdid && (h.sdid != edbn)) {
            // !!! Samples drop !!!
        }

        uint16_t* p = &m_data[0].data()[h.uwoffset];

        for (int i = 0; i < h.dc / 3; i++) {
            int ch;

            ch = group * 4 + ((*p >> 1) & 0x3);

            if (ch >= channels) {
                p += 3;
                continue;
            }

            if (pos[ch] >= maxsamples) {
                p += 3;
                continue;
            }

            uint32_t s = 0;

            s |= uint32_t((*(p++) >> 3) & 0x3F) << 4;
            s |= uint32_t(*(p++) & 0x1FF) << 10;
            s |= uint32_t(*(p++) & 0x1F) << 19;

            pos[ch]++;
        }
        dbn[group] = h.sdid;
    }
    // TODO: return all data counts
    return pos[0];
}
