#pragma once

#include <array>
#include <cstdint>
#include <linux/videodev2.h>
#include <memory>
#include <string>
#include <vector>

//! Small C++ wrapper around V4L2 device file descriptor
class V4L2Device {
public:
    static constexpr int MAX_PLANES = 3;

    //! Device buffer wrapper
    class Buffer : public std::enable_shared_from_this<Buffer> {
    public:
        virtual ~Buffer();
        Buffer(const Buffer& other) = delete;
        Buffer& operator=(const Buffer&) = delete;

        int numPlanes() const;
        uint8_t* data(unsigned int plane = 0) const;
        size_t length(unsigned int plane = 0) const;
        size_t bytesused(unsigned int plane = 0) const;
        v4l2_buffer& v4l2buf();

    private:
        friend class V4L2Device;

        Buffer(int fd, int type, int index);
        void update(const v4l2_buffer& buf);

        //! V4L2 device handle
        int m_fd;
        //! V4L2 buffer struct
        v4l2_buffer m_v4l2buf;
        //! V4L2 planes struct
        std::vector<v4l2_plane> m_v4l2planes;
        //! Buffer mmaped data
        std::vector<uint8_t*> m_data;
        //! Buffer size
        std::vector<size_t> m_length;
        //! _MPLANE API
        bool m_mplane = false;
    };
    using BufferPtr = std::shared_ptr<Buffer>;

    //! Create and open V4L2 device from path (/dev/videoX or /dev/vbiX)
    V4L2Device(std::string& path);
    virtual ~V4L2Device();

    //! Check device is open
    bool isOpen() const;
    //! There was error during last operation
    bool isError() const;
    //! Error code
    int error() const;
    //! Error description
    std::string errorString() const;
    //! File handle
    int fd() const;

    //! Device type
    v4l2_buf_type type() const;
    //! Device name
    std::string name() const;

    //! Call ioctl on device
    int ioctl(int ctl, void* param);

    //! Set format
    v4l2_format setFormat(const v4l2_format& fmt);
    //! Get format
    v4l2_format getFormat();

    //! Request buffers from driver
    void requestBuffers(int count);
    //! Free all allocated buffers
    void freeBuffers();
    //! Get buffer
    BufferPtr buffer(int n) const;
    //! Get all buffers
    const std::vector<BufferPtr>& buffers() const;

    //! Get buffer from hardware
    BufferPtr dequeueBuffer();
    //! Pass buffer to hardware
    bool queueBuffer(BufferPtr buffer);

private:
    V4L2Device(V4L2Device const&) = delete;
    V4L2Device& operator=(V4L2Device const&) = delete;

    int m_fd = -1;
    v4l2_capability m_caps;
    int m_lastError = 0;

    std::vector<BufferPtr> m_buffers;
    bool m_mplane = false;
};
