#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <poll.h>

/*!
 * \dir ./
 * This example shows how to use V4L2 events to get hardware timestamps
 */

namespace {
static const int BUFFER_COUNT = 8;
}

static bool shouldStop = false;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

void mainLoop(V4L2Device& inDev, V4L2Device& outDev)
{
    v4l2_event_subscription evts;
    memset(&evts, 0, sizeof(evts));
    evts.type = V4L2_EVENT_FORWARD_TIMESTAMP;

    // Subscribe to event
    inDev.ioctl(VIDIOC_SUBSCRIBE_EVENT, &evts);
    outDev.ioctl(VIDIOC_SUBSCRIBE_EVENT, &evts);

    pollfd pollfd[2];
    pollfd[0].fd = inDev.fd();
    pollfd[0].events = POLLPRI;
    pollfd[1].fd = outDev.fd();
    pollfd[1].events = POLLPRI;

    uint32_t eof_times[2] = { 0, 0 };

    while (!shouldStop) {
        // Wait for event
        poll(pollfd, 2, 100);

        if ((pollfd[0].revents & (POLLERR | POLLHUP | POLLNVAL))
            || (pollfd[1].revents & (POLLERR | POLLHUP | POLLNVAL)))
            break;

        // Check events
        for (int i = 0; i < 2; i++) {
            if (!(pollfd[i].revents & POLLPRI))
                continue;

            struct v4l2_event event = { 0 };
            struct v4l2_event_forward_timestamp ts_data;

            if (i == 0)
                inDev.ioctl(VIDIOC_DQEVENT, &event);
            else
                outDev.ioctl(VIDIOC_DQEVENT, &event);

            if (event.type != V4L2_EVENT_FORWARD_TIMESTAMP)
                continue;

            ts_data = *((v4l2_event_forward_timestamp*)&event.u.data);

            if (i == 0)
                std::cout << "IN  EVENT";
            else
                std::cout << "OUT EVENT";

            std::cout << " evt.sequence = " << event.sequence << " evt.timestamp = "
                      << event.timestamp.tv_sec * 1000000000ULL + event.timestamp.tv_nsec
                      << " buffer_sequence = " << ts_data.buffer_sequence
                      << " field = " << ts_data.field
                      << " eof_timestamp = " << ts_data.eof_timestamp
                      << " irq_timestamp = " << ts_data.irq_timestamp
                      << " field_size = " << ts_data.eof_timestamp - eof_times[i] << std::endl;
            eof_times[i] = ts_data.eof_timestamp;
        }
    }
}

int main(int argc, char** argv)
{
    std::string inPath, outPath;
    if (argc != 3) {
        std::cout << "Usage: " << argv[0] << " in_device out_device" << std::endl;
        return 0;
    }

    inPath = argv[1];
    std::cout << "Openning " << inPath << std::endl;
    V4L2Device inDev(inPath);

    if (!inDev.isOpen()) {
        std::cerr << "Cannot open device: " << inDev.errorString() << std::endl;
        return -1;
    }

    if (inDev.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device (" << inDev.name() << ") is not an input device" << std::endl;
        return -1;
    }

    outPath = argv[2];
    std::cout << "Openning " << outPath << std::endl;
    V4L2Device outDev(outPath);

    if (!outDev.isOpen()) {
        std::cerr << "Cannot open device: " << outDev.errorString() << std::endl;
        return -1;
    }

    if (outDev.type() != V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE) {
        std::cerr << "Device (" << outDev.name() << ") is not an input device" << std::endl;
        return -1;
    }

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    mainLoop(inDev, outDev);

    return 0;
}
