#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"

#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <poll.h>
#include <queue>

/*!
 * \dir ./
 * This example shows simple video passthrough example
 */

namespace {
static const int OUT_BUFFER_COUNT = 4;
static const int IN_BUFFER_COUNT = 8;
}

static bool shouldStop = false;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

v4l2_format setupFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;

    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;
    // 8-bit YUYV
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_YUYV;
    // 10-bit v210
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_V210;
    // Raw - whole SDI frame, with HANC, VANC, 10-bit dense packed
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    return fmt;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers,
    const v4l2_pix_format_mplane& fmt)
{
    // Fill playback queue
    for (auto buf : buffers) {
        if ((buf->v4l2buf().type == V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE)
            && (fmt.field == V4L2_FIELD_ALTERNATE))
            buf->v4l2buf().field = (buf->v4l2buf().index % 2) ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;

        dev.queueBuffer(buf);
    }
}

void mainLoop(V4L2Device& in, V4L2Device& out, const v4l2_pix_format_mplane& fmt)
{
    int inType = in.type();
    int outType = out.type();

    enum v4l2_field lastOutField;

    // Start capture
    in.ioctl(VIDIOC_STREAMON, &inType);
    // Start playback
    out.ioctl(VIDIOC_STREAMON, &outType);

    std::queue<V4L2Device::BufferPtr> inQueue, outQueue;

    pollfd pollfd[2];
    pollfd[0].fd = in.fd();
    pollfd[0].events = POLLIN;
    pollfd[1].fd = out.fd();
    pollfd[1].events = POLLOUT;

    while (!shouldStop) {
        // Wait for frame
        poll(pollfd, 2, 100);

        if (pollfd[0].revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        if (pollfd[1].revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);

        // Buffer received
        if (pollfd[0].revents & POLLIN) {
            // Get in buffer from driver queue
            V4L2Device::BufferPtr buf = in.dequeueBuffer();

            // Place it into our queue
            inQueue.push(buf);

            std::cout << "Received buffer at "
                      << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                      << "ns, id = " << buf->v4l2buf().index
                      << ", sequence = " << buf->v4l2buf().sequence
                      << ", field = " << buf->v4l2buf().field << ", timestamp = "
                      << (uint64_t)buf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                    + buf->v4l2buf().timestamp.tv_usec * 1000
                      << "ns, size = " << buf->bytesused(0) << std::endl;
        }

        // Buffer sended
        if (pollfd[1].revents & POLLOUT) {
            // Get sended buffer from queue
            V4L2Device::BufferPtr buf = out.dequeueBuffer();

            // Place it into our queue
            outQueue.push(buf);

            std::cout << "Sended buffer at "
                      << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                      << "ns, id = " << buf->v4l2buf().index
                      << ", sequence = " << buf->v4l2buf().sequence
                      << ", field = " << buf->v4l2buf().field << ", timestamp = "
                      << (uint64_t)buf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                    + buf->v4l2buf().timestamp.tv_usec * 1000
                      << "ns, size = " << buf->bytesused(0) << std::endl;
        }

        // Has free output and free input buffers
        if (!inQueue.empty() && !outQueue.empty()) {
            V4L2Device::BufferPtr outBuf = outQueue.front();
            outQueue.pop();

            V4L2Device::BufferPtr inBuf = inQueue.front();
            inQueue.pop();

            /*
             * PROCESS BUFFER DATA HERE
             */
            // Copy input to output
            outBuf->v4l2buf().field = inBuf->v4l2buf().field;
            lastOutField = (enum v4l2_field)inBuf->v4l2buf().field;
            memcpy(outBuf->data(0), inBuf->data(0), std::min(outBuf->length(0), outBuf->length(1)));

            std::cout << "Sending buffers at "
                      << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                      << "ns, in id = " << inBuf->v4l2buf().index << ", "
                      << "out id = " << outBuf->v4l2buf().index << ", inQueue = " << inQueue.size()
                      << ", outQueue = " << outQueue.size() << std::endl;

            out.queueBuffer(outBuf);
            in.queueBuffer(inBuf);
        }

        if (inQueue.size() >= IN_BUFFER_COUNT / 2) {
            // Input is faster or there is frame drops on output, flush input queue
            std::cout << "Input queue overflow at "
                      << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                      << "ns, flushing" << std::endl;
            while (!inQueue.empty()) {
                V4L2Device::BufferPtr inBuf = inQueue.front();
                in.queueBuffer(inBuf);
                inQueue.pop();
            }
        }

        if (outQueue.size() >= OUT_BUFFER_COUNT / 2) {
            // Output is faster or there is frame drops on input, flush output queue
            std::cout << "Output queue overflow at "
                      << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                      << "ns, flushing" << std::endl;
            while (!outQueue.empty()) {
                V4L2Device::BufferPtr outBuf = outQueue.front();
                outBuf->v4l2buf().field = (fmt.field == V4L2_FIELD_ALTERNATE)
                    ? ((lastOutField == V4L2_FIELD_TOP) ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP)
                    : fmt.field;
                lastOutField = (enum v4l2_field)outBuf->v4l2buf().field;
                out.queueBuffer(outBuf);
                outQueue.pop();
            }
        }
    }

    // Stop playback
    out.ioctl(VIDIOC_STREAMOFF, &outType);
    // Stop capture
    in.ioctl(VIDIOC_STREAMOFF, &inType);
}

int main(int argc, char** argv)
{
    std::string inDevPath, outDevPath;
    if (argc != 3) {
        std::cout << "Usage: " << argv[0] << " in_device out_device" << std::endl;
        return 0;
    }

    inDevPath = argv[1];
    std::cout << "Openning " << inDevPath << std::endl;
    V4L2Device in(inDevPath);

    if (!in.isOpen()) {
        std::cerr << "Cannot open input device: " << in.errorString() << std::endl;
        return -1;
    }

    if (in.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device (" << in.name() << ") is not an input device" << std::endl;
        return -1;
    }

    outDevPath = argv[2];
    std::cout << "Openning " << outDevPath << std::endl;
    V4L2Device out(outDevPath);

    if (!out.isOpen()) {
        std::cerr << "Cannot open output device: " << out.errorString() << std::endl;
        return -1;
    }

    if (out.type() != V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE) {
        std::cerr << "Device (" << out.name() << ") is not an output device" << std::endl;
        return -1;
    }

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format format;

    setupFormat(out);
    format = setupFormat(in);

    in.requestBuffers(IN_BUFFER_COUNT);
    out.requestBuffers(OUT_BUFFER_COUNT);

    std::cout << "Streaming with " << in.buffers().size() << " input and " << out.buffers().size()
              << " buffers" << std::endl;

    preroll(in, in.buffers(), format.fmt.pix_mp);
    preroll(out, out.buffers(), format.fmt.pix_mp);

    mainLoop(in, out, format.fmt.pix_mp);

    in.freeBuffers();
    out.freeBuffers();

    return 0;
}
