#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <poll.h>

/*!
 * \dir ./
 * This example shows simple video playback example
 */

namespace {
static const int BUFFER_COUNT = 4;
}

static bool shouldStop = false;
uint8_t* colorbar;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

v4l2_format setupFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;

    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;
    // 8-bit YUYV
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_YUYV;
    // 10-bit v210
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_V210;
    // Raw - whole SDI frame, with HANC, VANC, 10-bit dense packed
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Playback frame format is " << fmt << std::endl;

    return fmt;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers,
    const v4l2_pix_format_mplane& fmt)
{
    bool bottomField = false;
    // Fill playback queue
    for (auto buf : buffers) {
        if (fmt.field == V4L2_FIELD_ALTERNATE)
            buf->v4l2buf().field = bottomField ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;
        else
            buf->v4l2buf().field = fmt.field;

        dev.queueBuffer(buf);
        bottomField = !bottomField;
    }
}

void mainLoop(V4L2Device& dev, const v4l2_pix_format_mplane& fmt)
{
    int type = dev.type();

    bool bottomField = false;

    // Start playback
    dev.ioctl(VIDIOC_STREAMON, &type);

    pollfd pollfd;
    pollfd.fd = dev.fd();
    pollfd.events = POLLOUT;

    while (!shouldStop) {
        // Wait for frame
        poll(&pollfd, 1, 100);

        if (pollfd.revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Timeout, no frame
        if (!(pollfd.revents & POLLOUT))
            continue;

        // Get buffer from queue
        V4L2Device::BufferPtr buf = dev.dequeueBuffer();

        /*
         * PROCESS BUFFER DATA HERE
         */

        memcpy(buf->data(), colorbar, fmt.plane_fmt[0].sizeimage);
        // Don't forget to set correct field for V4L2_FIELD_ALTERNATE mode
        if (fmt.field == V4L2_FIELD_ALTERNATE) {
            buf->v4l2buf().field = bottomField ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;
            bottomField = !bottomField;
        } else {
            buf->v4l2buf().field = fmt.field;
        }

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);

        std::cout << "Sending buffer at "
                  << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                  << "ns, id = " << buf->v4l2buf().index
                  << ", sequence = " << buf->v4l2buf().sequence
                  << ", field = " << buf->v4l2buf().field << ", timestamp = "
                  << (uint64_t)buf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + buf->v4l2buf().timestamp.tv_usec * 1000
                  << "ns, size = " << buf->bytesused(0) << " " << (uint64_t)buf->data()
                  << std::endl;

        // Return buffer back to queue
        dev.queueBuffer(buf);
    }

    // Stop playback
    dev.ioctl(VIDIOC_STREAMOFF, &type);
}

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc != 2) {
        std::cout << "Usage: " << argv[0] << " device" << std::endl;
        return 0;
    }

    devPath = argv[1];
    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    if (dev.type() != V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE) {
        std::cerr << "Device (" << dev.name() << ") is not an output device" << std::endl;
        return -1;
    }

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format format;

    format = setupFormat(dev);
    dev.requestBuffers(BUFFER_COUNT);

    colorbar = new uint8_t[format.fmt.pix_mp.plane_fmt[0].sizeimage];
    generateRP219ColorBarUYVY(
        colorbar, format.fmt.pix_mp.pixelformat, format.fmt.pix_mp.width, format.fmt.pix_mp.height);

    std::cout << "Playback with " << dev.buffers().size() << " buffers" << std::endl;

    preroll(dev, dev.buffers(), format.fmt.pix_mp);
    mainLoop(dev, format.fmt.pix_mp);

    dev.freeBuffers();

    return 0;
}
