#include "ALSADevice.hpp"
#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <climits>
#include <cmath>
#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <list>
#include <poll.h>
#include <tuple>

/*!
 * \dir ./
 * This example shows video + vbi + audio capture example
 */

namespace {
static const int BUFFER_COUNT = 8;
// ALSA chunk size, should be less than frame (field) period
static const int AUDIO_PERIOD_SIZE_US = 10000;
static const int AUDIO_BUFFER_SIZE_US = 1000000;
static const int AUDIO_CHANNELS = 2;
static const snd_pcm_uframes_t AUDIO_SAMPLE_RATE = 48000;
}

static bool shouldStop = false;
static uint8_t* colorbar;
static uint8_t* white;
snd_pcm_uframes_t audioBufSize;
int32_t* audioBufSilence;
int32_t* audioBufSin;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

// VBI should be explicit enabled by controls, because it requires some overhead in driver
void enableVBI(V4L2Device& dev)
{
    struct v4l2_control ctl;
    ctl.id = V4L2_CID_FORWARD_ENABLE_VANC;
    ctl.value = true;

    if (dev.ioctl(VIDIOC_S_CTRL, &ctl))
        std::cerr << "Cannot enable VANC: " << dev.errorString() << std::endl;
}

//! Get required audio samples per video frame
snd_pcm_uframes_t getAudioPerVideoFrame(V4L2Device& dev, const v4l2_pix_format_mplane& fmt)
{
    v4l2_dv_timings tim;
    if (dev.ioctl(VIDIOC_G_DV_TIMINGS, &tim))
        return 0;

    uint64_t denom = tim.bt.pixelclock;
    uint64_t num
        = AUDIO_SAMPLE_RATE * V4L2_DV_BT_FRAME_WIDTH(&tim.bt) * V4L2_DV_BT_FRAME_HEIGHT(&tim.bt);

    // FIXME: No /1.001 compensation
    if (tim.bt.interlaced && (fmt.field == V4L2_FIELD_ALTERNATE))
        denom *= 2;

    return num / denom;
}

v4l2_format setupVideoFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;
    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;
    // 8-bit YUYV
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_YUYV;
    // 10-bit v210
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_V210;
    // Raw - whole SDI frame, with HANC, VANC, 10-bit dense packed
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Playback frame format is " << fmt << std::endl;

    return fmt;
}

v4l2_format setupVBIFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // 10-bit data in 16-bit words, interleaved CYCYCYC as in SDI
    fmt.fmt.vbi.sample_format = V4L2_PIX_FMT_Y10;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Playback VBI format is " << fmt << std::endl;

    return fmt;
}

void preroll(
    V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers, v4l2_pix_format_mplane fmt)
{
    // Fill capture queue
    for (auto buf : buffers) {
        if (fmt.field == V4L2_FIELD_ALTERNATE)
            buf->v4l2buf().field = (buf->v4l2buf().index % 2) ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;
        else
            buf->v4l2buf().field = fmt.field;

        if (dev.type() == V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE)
            memcpy(buf->data(), white, fmt.plane_fmt[0].sizeimage);

        dev.queueBuffer(buf);
    }
}

void prerollAudio(ALSADevice& dev, int audioBufSize, int buffers)
{
    int32_t* audio_buf = new int32_t[audioBufSize * AUDIO_CHANNELS];
    memset(audio_buf, 0, audioBufSize * AUDIO_CHANNELS * sizeof(int32_t));

    for (int i = 0; i < buffers; i++)
        snd_pcm_writei(dev.pcm().get(), audio_buf, audioBufSize);

    delete[] audio_buf;
}

void setupAudio(ALSADevice& dev)
{
    snd_pcm_hw_params_t* params = nullptr;
    snd_pcm_sw_params_t* sw_params = nullptr;
    snd_pcm_format_t format;
    snd_pcm_uframes_t period;
    unsigned int channels, rate;
    int dir;

    // Alloc hardware parameter structs
    snd_pcm_hw_params_malloc(&params);

    // Get default hardware parameters
    snd_pcm_hw_params_any(dev.pcm().get(), params);

    // Interleave channels
    snd_pcm_hw_params_set_access(dev.pcm().get(), params, SND_PCM_ACCESS_RW_INTERLEAVED);

    // Set period and buffer sizes
    snd_pcm_hw_params_set_period_time(dev.pcm().get(), params, AUDIO_PERIOD_SIZE_US, 0);
    snd_pcm_hw_params_set_buffer_time(dev.pcm().get(), params, AUDIO_BUFFER_SIZE_US, 0);

    // Set number of channels
    snd_pcm_hw_params_set_channels(dev.pcm().get(), params, AUDIO_CHANNELS);

    // Set hardware parameters, you can also change format, channels and other params prior this
    // call
    snd_pcm_hw_params(dev.pcm().get(), params);

    // Extract accepted parameters,
    snd_pcm_hw_params_get_format(params, &format);
    snd_pcm_hw_params_get_channels(params, &channels);
    snd_pcm_hw_params_get_rate(params, &rate, &dir);
    snd_pcm_hw_params_get_period_size(params, &period, &dir);

    snd_pcm_hw_params_free(params);

    // Alloc software parameter structs
    snd_pcm_sw_params_alloca(&sw_params);
    // Get current software parameters
    snd_pcm_sw_params_current(dev.pcm().get(), sw_params);
    // Enable ALSA timestamping so we can always sync audio with video
    snd_pcm_sw_params_set_tstamp_mode(dev.pcm().get(), sw_params, SND_PCM_TSTAMP_ENABLE);
    // Disable ALSA auto-start
    snd_pcm_sw_params_set_start_threshold(dev.pcm().get(), sw_params, LONG_MAX);
    // Set software parameters
    snd_pcm_sw_params(dev.pcm().get(), sw_params);

    snd_pcm_prepare(dev.pcm().get());

    std::cout << "ALSA playback " << channels << " x " << snd_pcm_format_name(format) << " x "
              << rate << " with period of " << period << " samples" << std::endl;
}

void mainLoop(V4L2Device& videoDev, V4L2Device& vbiDev, ALSADevice& audioDev,
    const v4l2_pix_format_mplane& videoFormat, const v4l2_vbi_format& vbiFormat)
{
    int videoType = videoDev.type();
    int vbiType = vbiDev.type();

    bool bottomField = false;
    int num = 0;

    // Alloc and setup ALSA status structure - used to get current ALSA status
    std::shared_ptr<snd_pcm_status_t> status;
    snd_pcm_status_t* sp;
    snd_pcm_status_malloc(&sp);
    status = std::shared_ptr<snd_pcm_status_t>(sp, snd_pcm_status_free);

    // Forward-specific setup - this will tell ALSA and driver to return real timestamp in status
    // This timestamp will be same as in corresponding video buffer
    snd_pcm_audio_tstamp_config_t config;
    config.report_delay = 0;
    config.type_requested = SND_PCM_AUDIO_TSTAMP_TYPE_LINK_ABSOLUTE;
    snd_pcm_status_set_audio_htstamp_config(status.get(), &config);

    // Start capture, VBI device first!
    vbiDev.ioctl(VIDIOC_STREAMON, &vbiType);
    videoDev.ioctl(VIDIOC_STREAMON, &videoType);
    // Start audio capture
    snd_pcm_start(audioDev.pcm().get());

    pollfd pollfd[1];

    pollfd[0].fd = videoDev.fd();
    pollfd[0].events = POLLOUT;

    while (!shouldStop) {
        // Wait for frame
        poll(pollfd, 1, 100);

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);
        uint64_t curTs = (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec;

        if (pollfd[0].revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Video/VBI frame sended
        if (pollfd[0].revents & POLLOUT) {
            // Get buffer from driver, video first.
            // Video and VBI is always in sync if number of queued buffers are always same
            // To be sure, you can check sequence fields
            V4L2Device::BufferPtr videoBuf = videoDev.dequeueBuffer();
            V4L2Device::BufferPtr vbiBuf = vbiDev.dequeueBuffer();

            uint64_t vidTs = (uint64_t)videoBuf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + videoBuf->v4l2buf().timestamp.tv_usec * 1000;
            uint64_t vbiTs = (uint64_t)vbiBuf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + vbiBuf->v4l2buf().timestamp.tv_usec * 1000;

            std::cout << "Sended video buffer at " << curTs << "ns, "
                      << "video id = " << videoBuf->v4l2buf().index
                      << ", video sequence = " << videoBuf->v4l2buf().sequence
                      << ", video field = " << videoBuf->v4l2buf().field
                      << ", video timestamp = " << vidTs
                      << "ns, video size = " << videoBuf->v4l2buf().bytesused
                      << ", vbi id = " << vbiBuf->v4l2buf().index
                      << ", vbi sequence = " << vbiBuf->v4l2buf().sequence
                      << ", vbi field = " << vbiBuf->v4l2buf().field
                      << ", vbi timestamp = " << vbiTs
                      << "ns, vbi size = " << vbiBuf->v4l2buf().bytesused << std::endl;

            // Get current ALSA status
            snd_pcm_status(audioDev.pcm().get(), status.get());
            // .. and timestamp
            snd_htimestamp_t ts;
            snd_pcm_status_get_audio_htstamp(status.get(), &ts);
            // Timestamp, drop ns part as in V4L2
            uint64_t audioTs = (uint64_t)ts.tv_sec * 1000000000ULL + (ts.tv_nsec / 1000) * 1000;
            snd_pcm_sframes_t audioBuffered = snd_pcm_status_get_delay(status.get());

            /*
             *  PREPARE DATA HERE
             */

            memcpy(videoBuf->data(), ((num % 8) < 1) ? colorbar : white,
                videoFormat.plane_fmt[0].sizeimage);

            // Don't forget to set correct field for V4L2_FIELD_ALTERNATE mode
            if (videoFormat.field == V4L2_FIELD_ALTERNATE) {
                videoBuf->v4l2buf().field = bottomField ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;
                vbiBuf->v4l2buf().field = bottomField ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;
                bottomField = !bottomField;
            }

            snd_pcm_sframes_t size = snd_pcm_writei(audioDev.pcm().get(),
                ((num % 8) < 1) ? audioBufSin : audioBufSilence, audioBufSize);

            std::cout << "Sending " << size << " audio frames at " << curTs << "ns, "
                      << "audio timestamp = " << audioTs << ", buffered = " << audioBuffered
                      << std::endl;

            // Return video/vbi buffers back to queue
            vbiDev.queueBuffer(vbiBuf);
            videoDev.queueBuffer(videoBuf);

            num++;
        }
    }

    // Stop audio capture
    snd_pcm_drop(audioDev.pcm().get());
    // Stop capture, Video device first!
    videoDev.ioctl(VIDIOC_STREAMOFF, &videoType);
    vbiDev.ioctl(VIDIOC_STREAMOFF, &vbiType);
}

int main(int argc, char** argv)
{
    std::string videoPath, vbiPath, audioPath;
    if (argc != 4) {
        std::cout << "Usage: " << argv[0] << " video_device vbi_device audio_device" << std::endl;
        return 0;
    }

    videoPath = argv[1];
    std::cout << "Openning " << videoPath << std::endl;
    V4L2Device videoDev(videoPath);

    if (!videoDev.isOpen()) {
        std::cerr << "Cannot open video device: " << videoDev.errorString() << std::endl;
        return -1;
    }

    if (videoDev.type() != V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE) {
        std::cerr << "Device " << videoPath << " is not an output video device" << std::endl;
        return -1;
    }

    vbiPath = argv[2];
    std::cout << "Openning " << vbiPath << std::endl;
    V4L2Device vbiDev(vbiPath);

    if (!vbiDev.isOpen()) {
        std::cerr << "Cannot open VBI device: " << vbiDev.errorString() << std::endl;
        return -1;
    }

    if (vbiDev.type() != V4L2_BUF_TYPE_VBI_OUTPUT) {
        std::cerr << "Device " << vbiPath << " is not an output VBI device" << std::endl;
        return -1;
    }

    audioPath = argv[3];
    ALSADevice audioDev(audioPath, SND_PCM_STREAM_PLAYBACK);

    if (!audioDev.isOpen()) {
        std::cerr << "Cannot open audio device: " << audioDev.errorString() << std::endl;
        return -1;
    }

    setupAudio(audioDev);

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format videoFormat, vbiFormat;

    enableVBI(videoDev);

    videoFormat = setupVideoFormat(videoDev);
    vbiFormat = setupVBIFormat(vbiDev);

    videoDev.requestBuffers(BUFFER_COUNT);
    vbiDev.requestBuffers(BUFFER_COUNT);

    colorbar = new uint8_t[videoFormat.fmt.pix_mp.plane_fmt[0].sizeimage];
    generateRP219ColorBarUYVY(colorbar, videoFormat.fmt.pix_mp.pixelformat,
        videoFormat.fmt.pix_mp.width, videoFormat.fmt.pix_mp.height);
    white = new uint8_t[videoFormat.fmt.pix_mp.plane_fmt[0].sizeimage];
    for (int i = 0; i < videoFormat.fmt.pix_mp.plane_fmt[0].sizeimage / 4; i++)
        ((uint32_t*)white)[i] = 0xF080F080;

    // Alloc audio buffer
    audioBufSize = getAudioPerVideoFrame(videoDev, videoFormat.fmt.pix_mp);
    audioBufSilence = new int32_t[audioBufSize * AUDIO_CHANNELS];
    audioBufSin = new int32_t[audioBufSize * AUDIO_CHANNELS];
    for (snd_pcm_uframes_t i = 0; i < audioBufSize; i++) {
        audioBufSilence[i * 2 + 0] = 0;
        audioBufSilence[i * 2 + 1] = 0;
        audioBufSin[i * 2 + 0]
            = sin((float)i / (audioBufSize / 10.0f) * 2.0f * M_PI) * (float)(1 << 30);
        audioBufSin[i * 2 + 1]
            = sin((float)i / (audioBufSize / 10.0f) * 2.0f * M_PI) * (float)(1 << 30);
    }

    std::cout << "Playback with " << videoDev.buffers().size() << " buffers" << std::endl;

    // Preroll video
    preroll(vbiDev, vbiDev.buffers(), videoFormat.fmt.pix_mp);
    preroll(videoDev, videoDev.buffers(), videoFormat.fmt.pix_mp);
    prerollAudio(audioDev, getAudioPerVideoFrame(videoDev, videoFormat.fmt.pix_mp),
        videoDev.buffers().size());

    mainLoop(videoDev, vbiDev, audioDev, videoFormat.fmt.pix_mp, vbiFormat.fmt.vbi);

    delete[] audioBufSilence;
    delete[] audioBufSin;
    delete[] colorbar;
    delete[] white;

    videoDev.freeBuffers();
    vbiDev.freeBuffers();

    return 0;
}
