#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"

#include <iomanip>
#include <iostream>

/*!
 * \dir ./
 * This example shows how to work with timings (video modes) for device
 */

/*!
 * \brief List all supported timings
 */
void enumTimings(V4L2Device& dev)
{
    std::cout << "Supported device timigs: " << std::endl;

    v4l2_enum_dv_timings e = { 0 };
    e.index = 0;

    // Request timings with incrementing index until driver returns -1
    while (!dev.ioctl(VIDIOC_ENUM_DV_TIMINGS, &e)) {
        v4l2_dv_timings tim = e.timings;
        std::cout << "\t" << tim << std::endl;
        e.index++;
    }
}

/*!
 * \brief Get current device timings
 * For input device, this is NOT a timings on input - see \ref queryTimings()
 */
void getTimings(V4L2Device& dev)
{
    v4l2_dv_timings tim;
    if (dev.ioctl(VIDIOC_G_DV_TIMINGS, &tim)) {
        std::cerr << "Cannot get device timigs: " << dev.errorString() << std::endl;
        return;
    }
    std::cout << "Current device timigs: " << tim << std::endl;
}

/*!
 * \brief Query current timings on device input
 * This is a timings on input
 */
void queryTimings(V4L2Device& dev)
{
    v4l2_dv_timings tim;
    if (dev.ioctl(VIDIOC_QUERY_DV_TIMINGS, &tim)) {
        if (dev.error() == ENOLINK)
            std::cout << "No signal on input" << std::endl;
        else if (dev.error() == ERANGE)
            std::cout << "Unsupported signal on input" << std::endl;
        else
            std::cerr << "Cannot query device timigs: " << dev.errorString() << std::endl;
        return;
    }

    std::cout << "Current timigs on input: " << tim << std::endl;
}

/*!
 * \brief Set current device timings
 */
void setTimings(V4L2Device& dev)
{
    // Query current timing on input
    v4l2_dv_timings tim;
    if (dev.ioctl(VIDIOC_QUERY_DV_TIMINGS, &tim)) {
        std::cerr << "Cannot query device timigs: " << dev.errorString() << std::endl;
        return;
    }
    // Set timings on device, so timings input == device timings
    if (dev.ioctl(VIDIOC_S_DV_TIMINGS, &tim)) {
        std::cerr << "Cannot set device timigs: " << dev.errorString() << std::endl;
        return;
    }

    std::cout << "Device timigs set to: " << tim << std::endl;
}

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc > 1) {
        devPath = argv[1];
    } else {
        std::vector<Board::Info> boards = Board::enumerate();
        if (!boards.size() || !boards[0].videoDevs.size() || boards[0].videoDevs[0].empty()) {
            std::cerr << "Cannot find any device!" << std::endl;
            return -1;
        }
        devPath = boards[0].videoDevs[0];
    }

    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    if (dev.type() == V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE)
        std::cout << "Device (" << dev.name() << ") is video capture" << std::endl;
    else if (dev.type() == V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE)
        std::cout << "Device (" << dev.name() << ") is video output" << std::endl;
    else {
        std::cerr << "Device (" << dev.name() << ") has wrong type (" << dev.type() << ")"
                  << std::endl;
        return -1;
    }

    enumTimings(dev);
    getTimings(dev);
    queryTimings(dev);
    setTimings(dev);

    return 0;
}
