/*
   forward-alsa-fd720.c - ALSA driver for SoftLab-NSK FD720 video board

   Copyright (C) 2017 - 2024 SoftLab-NSK <forward@softlab.tv>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "forward-alsa.h"
#include "forward-alsa-io.h"
#include "forward-vdma.h"

#include "fd720_reg.h"

#define FD720_IO_IRQ_NUM(io) ((io->index) * 3)
#define FD720_BUFFER_NEXT(buf, offset, count) ((((unsigned int)(buf)) + (offset)) % (count))
#define FD720_NUMBER_IN_BUFFERS (4)

static forward_irq_t fd720_irq_mask(const struct forward_alsa_io *io)
{
	forward_irq_t result = { 0 };
	forward_irq_flags_set_one(result, FD720_IO_IRQ_NUM(io));
	return result;
}

static bool fd720_irq_info(const struct forward_alsa_io *io, const forward_irq_t *irq, int *offset,
			   size_t *size, bool *field)
{
	int buf;
	FD720_VideoInANCCounter anc;

	if (!forward_irq_has_irq(*irq, FD720_IO_IRQ_NUM(io)))
		return false;

	anc = FD720_VideoInANCCounter_R(io->alsa->dev->csr, io->index);

	buf = forward_irq_extract_data(*irq, FD720_IO_IRQ_NUM(io) + 1, 2);
	buf = FD720_BUFFER_NEXT(buf, -1, FD720_NUMBER_IN_BUFFERS);

	*offset = buf * 0x00200000;
	*size = anc.counter;
	*field = io->interlaced ? (buf & 0x1) : false;

	return true;
}

static void fd720_get_region(const struct forward_alsa_io *io, u32 *address, size_t *size)
{
	*address = io->index * 0x20000000UL + 0x10000000UL;
	*size = 0x00800000;
	forward_vdma_get_region(io->alsa->dev->vdma, io, *address, *size);
}

static void fd720_put_region(const struct forward_alsa_io *io)
{
	forward_vdma_put_all_regions(io->alsa->dev->vdma, io);
}

static void fd720_fill_hw(struct forward_alsa_io *io, struct snd_pcm_hardware *hw)
{
	FD720_VideoInInfoFrame inf = FD720_VideoInInfoFrame_R(io->alsa->dev->csr, io->index);

	hw->formats = SNDRV_PCM_FMTBIT_S32_LE;
	hw->rates = SNDRV_PCM_RATE_32000 | SNDRV_PCM_RATE_44100 | SNDRV_PCM_RATE_48000 |
		    SNDRV_PCM_RATE_88200 | SNDRV_PCM_RATE_96000 | SNDRV_PCM_RATE_176400 |
		    SNDRV_PCM_RATE_192000;
	hw->rate_min = 32000;
	hw->rate_max = 96000;
	hw->channels_min = 0;
	hw->channels_max = 8;
	hw->buffer_bytes_max = 2 * 1024 * 1024;
	hw->period_bytes_min = 4 * 1024;
	hw->period_bytes_max = 128 * 1024;
	hw->periods_min = 4;
	hw->periods_max = 128;

	if (!inf.aiValid)
		return;

	switch (inf.aiSF) {
	case 1:
		hw->rates = SNDRV_PCM_RATE_32000;
		hw->rate_min = 32000;
		hw->rate_max = 32000;
		break;
	case 2:
		hw->rates = SNDRV_PCM_RATE_44100;
		hw->rate_min = 44100;
		hw->rate_max = 44100;
		break;
	case 3:
		hw->rates = SNDRV_PCM_RATE_48000;
		hw->rate_min = 48000;
		hw->rate_max = 48000;
		break;
	case 4:
		hw->rates = SNDRV_PCM_RATE_88200;
		hw->rate_min = 88200;
		hw->rate_max = 88200;
		break;
	case 5:
		hw->rates = SNDRV_PCM_RATE_96000;
		hw->rate_min = 96000;
		hw->rate_max = 96000;
		break;
	case 6:
		hw->rates = SNDRV_PCM_RATE_176400;
		hw->rate_min = 176400;
		hw->rate_max = 176400;
		break;
	case 7:
		hw->rates = SNDRV_PCM_RATE_192000;
		hw->rate_min = 192000;
		hw->rate_max = 192000;
		break;
	default:
		break;
	}
	if (inf.aiCC > 0) {
		hw->channels_min = inf.aiCC + 1;
		hw->channels_max = inf.aiCC + 1;
	}
}

static void fd720_update_format(struct forward_alsa_io *io)
{
	io->type = FORWARD_ALSA_HDMI;
	io->hw_format = SNDRV_PCM_FORMAT_S24;
	io->hw_channels = 0;
}

static int fd720_start(struct forward_alsa_io *io)
{
	struct forward_dev *dev = io->alsa->dev;
	dev->cfg.toggle_streaming(dev, io->index, true);
	return 0;
}

static void fd720_stop(struct forward_alsa_io *io)
{
	struct forward_dev *dev = io->alsa->dev;
	dev->cfg.toggle_streaming(dev, io->index, false);
}

struct forward_alsa_dev_ops fd720_alsa_ops = {
	.irq_mask = fd720_irq_mask,
	.irq_info = fd720_irq_info,
	.get_region = fd720_get_region,
	.put_region = fd720_put_region,
	.fill_hw = fd720_fill_hw,
	.update_format = fd720_update_format,
	.start = fd720_start,
	.stop = fd720_stop,
};
