/*
   forward-ethernet-ethtool.c - Ethernet driver for SoftLab-NSK Forward boards

   Copyright (C) 2017 - 2024 SoftLab-NSK <forward@softlab.tv>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "forward.h"
#include "forward-ethernet.h"
#include <linux/pci.h>
#include <linux/ethtool.h>

static void forward_get_drvinfo(struct net_device *dev, struct ethtool_drvinfo *drvinfo)
{
	struct forward_ethernet *eth = netdev_priv(dev);
	strscpy(drvinfo->driver, FORWARD_ETHERNET_MODULE_NAME, sizeof(drvinfo->driver));
	if (eth->dev->pci_dev)
		strscpy(drvinfo->bus_info, pci_name(eth->dev->pci_dev), sizeof(drvinfo->bus_info));
	strscpy(drvinfo->version, FORWARD_VERSION_STRING, sizeof(drvinfo->version));
	snprintf(drvinfo->fw_version, sizeof(drvinfo->fw_version), "%d",
		 eth->dev->fw_version & 0xFFFF);
}

static u32 forward_get_link(struct net_device *dev)
{
	struct forward_ethernet *eth = netdev_priv(dev);
	return eth->ops->get_link_status(eth);
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(6, 11, 0)) || \
	(RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(9, 6))
static int forward_get_ts_info(struct net_device *netdev, struct kernel_ethtool_ts_info *info)
#else
static int forward_get_ts_info(struct net_device *netdev, struct ethtool_ts_info *info)
#endif
{
	struct forward_ethernet *eth = netdev_priv(netdev);

	info->so_timestamping = SOF_TIMESTAMPING_TX_SOFTWARE | SOF_TIMESTAMPING_RX_SOFTWARE |
				SOF_TIMESTAMPING_SOFTWARE | SOF_TIMESTAMPING_TX_HARDWARE |
				SOF_TIMESTAMPING_RX_HARDWARE | SOF_TIMESTAMPING_RAW_HARDWARE;
	info->tx_types = (1 << HWTSTAMP_TX_OFF) | (1 << HWTSTAMP_TX_ON);
	info->rx_filters = (1 << HWTSTAMP_FILTER_NONE) | (1 << HWTSTAMP_FILTER_ALL);

	info->phc_index = ptp_clock_index(eth->board->ptp_clock);

	return 0;
}

static int forward_get_link_ksettings(struct net_device *netdev, struct ethtool_link_ksettings *cmd)
{
	struct forward_ethernet *eth = netdev_priv(netdev);
	eth->ops->get_link_ksettings(eth, cmd);
	return 0;
}

static int forward_set_link_ksettings(struct net_device *netdev,
				      const struct ethtool_link_ksettings *cmd)
{
	struct forward_ethernet *eth = netdev_priv(netdev);
	return eth->ops->set_link_ksettings(eth, cmd);
}

static int forward_get_module_info(struct net_device *netdev, struct ethtool_modinfo *modinfo)
{
	struct forward_ethernet *eth = netdev_priv(netdev);
	bool absense;
	u8 type;

	eth->ops->sfp_status(eth, &absense, NULL, NULL);

	if (absense)
		return -EIO;

	if (eth->ops->sfp_read_reg(eth, 0x00, &type))
		return -EIO;

	switch (type) {
	case 0x3:
		modinfo->type = ETH_MODULE_SFF_8472;
		modinfo->eeprom_len = ETH_MODULE_SFF_8472_LEN;
		break;
	default:
		return -EINVAL;
		break;
	}

	return 0;
}

static int forward_get_module_eeprom(struct net_device *netdev, struct ethtool_eeprom *ee, u8 *data)
{
	struct forward_ethernet *eth = netdev_priv(netdev);
	u16 addr;

	if ((ee->len == 0) || ((ee->offset + ee->len) > 512))
		return -EINVAL;

	memset(data, 0, ee->len);

	for (addr = 0; addr <= ee->len; addr++) {
		if (eth->ops->sfp_read_reg(eth, ee->offset + addr, &data[addr]))
			break;
	}
	return 0;
}

const struct ethtool_ops forward_ethtool_ops = {
	.get_drvinfo = forward_get_drvinfo,
	.get_link = forward_get_link,
	.get_link_ksettings = forward_get_link_ksettings,
	.set_link_ksettings = forward_set_link_ksettings,
	.get_ts_info = forward_get_ts_info,
	.get_module_info = forward_get_module_info,
	.get_module_eeprom = forward_get_module_eeprom,
};
