/*
   forward-v4l2-asi.c - DVB-ASI utilities for SoftLab-NSK Forward video boards

   Copyright (C) 2017 - 2024 SoftLab-NSK <forward@softlab.tv>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "forward.h"
#include "forward-vdma.h"
#include "forward-v4l2-asi.h"

#include <linux/vmalloc.h>

#define MAX_PCR ((1ULL << 33) * 300 - 1)
#define DEFAULT_BUFFER_SIZE (16384 * 188)
#define HW_BUFFER_SIZE_WORDS 540000
#define HW_BUFFER_SIZE ((((1000 * 270 * 20 / 8) - 1) / 4096 + 1) * 4096)
#define HW_BUFFER_COUNT 32

static const u16 t10to8[1024] = {
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0x00ef, 0x00e0, 0x00e7, 0xffff, 0x00f0, 0x00ff, 0x00eb, 0x00f8,
	0x00ed, 0x00ee, 0xffff, 0xffff, 0x00e1, 0x00e2, 0x00f3, 0x00e4, 0x00f5, 0x00f6, 0x00f7,
	0x00e8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0xffff, 0xffff, 0x00fe, 0x00fd,
	0x00e3, 0x00fb, 0x00e5, 0x00e6, 0x00e8, 0x00f7, 0x00e9, 0x00ea, 0x00e4, 0x00ec, 0x00e2,
	0x00e1, 0xffff, 0xffff, 0x00f1, 0x00f2, 0x00f8, 0x00f4, 0x00ff, 0x00f0, 0xffff, 0x00e7,
	0x00e0, 0x00ef, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0x000f, 0x0000, 0x0007, 0xffff, 0x0010, 0x001f, 0x000b, 0x0018, 0x000d, 0x000e,
	0xffff, 0xffff, 0x0001, 0x0002, 0x0013, 0x0004, 0x0015, 0x0016, 0x0017, 0x0008, 0x0019,
	0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0xffff, 0xffff, 0x001e, 0x001d, 0x0003, 0x001b,
	0x0005, 0x0006, 0x0008, 0x0017, 0x0009, 0x000a, 0x0004, 0x000c, 0x0002, 0x0001, 0xffff,
	0xffff, 0x0011, 0x0012, 0x0018, 0x0014, 0x001f, 0x0010, 0xffff, 0x0007, 0x0000, 0x000f,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x006f,
	0x0060, 0x0067, 0xffff, 0x0070, 0x007f, 0x006b, 0x0078, 0x006d, 0x006e, 0xffff, 0xffff,
	0x0061, 0x0062, 0x0073, 0x0064, 0x0075, 0x0076, 0x0077, 0x0068, 0x0079, 0x007a, 0x007b,
	0x007c, 0x007d, 0x007e, 0xffff, 0xffff, 0x007e, 0x007d, 0x0063, 0x007b, 0x0065, 0x0066,
	0x0068, 0x0077, 0x0069, 0x006a, 0x0064, 0x006c, 0x0062, 0x0061, 0xffff, 0xffff, 0x0071,
	0x0072, 0x0078, 0x0074, 0x007f, 0x0070, 0xffff, 0x0067, 0x0060, 0x006f, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x008f, 0x0080, 0x0087,
	0xffff, 0x0090, 0x009f, 0x008b, 0x0098, 0x008d, 0x008e, 0xffff, 0xffff, 0x0081, 0x0082,
	0x0093, 0x0084, 0x0095, 0x0096, 0x0097, 0x0088, 0x0099, 0x009a, 0x009b, 0x009c, 0x009d,
	0x009e, 0xffff, 0xffff, 0x009e, 0x009d, 0x0083, 0x009b, 0x0085, 0x0086, 0x0088, 0x0097,
	0x0089, 0x008a, 0x0084, 0x008c, 0x0082, 0x0081, 0xffff, 0xffff, 0x0091, 0x0092, 0x0098,
	0x0094, 0x009f, 0x0090, 0xffff, 0x0087, 0x0080, 0x008f, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x00af, 0x00a0, 0x00a7, 0xffff, 0x00b0,
	0x00bf, 0x00ab, 0x00b8, 0x00ad, 0x00ae, 0xffff, 0xffff, 0x00a1, 0x00a2, 0x00b3, 0x00a4,
	0x00b5, 0x00b6, 0x00b7, 0x00a8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0xffff,
	0xffff, 0x00be, 0x00bd, 0x00a3, 0x00bb, 0x00a5, 0x00a6, 0x00a8, 0x00b7, 0x00a9, 0x00aa,
	0x00a4, 0x00ac, 0x00a2, 0x00a1, 0xffff, 0xffff, 0x00b1, 0x00b2, 0x00b8, 0x00b4, 0x00bf,
	0x00b0, 0xffff, 0x00a7, 0x00a0, 0x00af, 0xffff, 0x01bc, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0x00cf, 0x00c0, 0x00c7, 0xffff, 0x00d0, 0x00df, 0x00cb,
	0x00d8, 0x00cd, 0x00ce, 0xffff, 0xffff, 0x00c1, 0x00c2, 0x00d3, 0x00c4, 0x00d5, 0x00d6,
	0x00d7, 0x00c8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0xffff, 0xffff, 0x00de,
	0x00dd, 0x00c3, 0x00db, 0x00c5, 0x00c6, 0x00c8, 0x00d7, 0x00c9, 0x00ca, 0x00c4, 0x00cc,
	0x00c2, 0x00c1, 0xffff, 0xffff, 0x00d1, 0x00d2, 0x00d8, 0x00d4, 0x00df, 0x00d0, 0xffff,
	0x00c7, 0x00c0, 0x00cf, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0x00ef, 0x00e0, 0x00e7, 0xffff, 0x00f0, 0x00ff, 0x00eb, 0x00f8, 0x00ed,
	0x00ee, 0xffff, 0xffff, 0x00e1, 0x00e2, 0x00f3, 0x00e4, 0x00f5, 0x00f6, 0x00f7, 0x00e8,
	0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0xffff, 0xffff, 0x00fe, 0x00fd, 0x00e3,
	0x00fb, 0x00e5, 0x00e6, 0x00e8, 0x00f7, 0x00e9, 0x00ea, 0x00e4, 0x00ec, 0x00e2, 0x00e1,
	0xffff, 0xffff, 0x00f1, 0x00f2, 0x00f8, 0x00f4, 0x00ff, 0x00f0, 0xffff, 0x00e7, 0x00e0,
	0x00ef, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0x00ef, 0x00e0, 0x00e7, 0xffff, 0x00f0, 0x00ff, 0x00eb, 0x00f8, 0x00ed, 0x00ee, 0xffff,
	0xffff, 0x00e1, 0x00e2, 0x00f3, 0x00e4, 0x00f5, 0x00f6, 0x00f7, 0x00e8, 0x00f9, 0x00fa,
	0x00fb, 0x00fc, 0x00fd, 0x00fe, 0xffff, 0xffff, 0x00fe, 0x00fd, 0x00e3, 0x00fb, 0x00e5,
	0x00e6, 0x00e8, 0x00f7, 0x00e9, 0x00ea, 0x00e4, 0x00ec, 0x00e2, 0x00e1, 0xffff, 0xffff,
	0x00f1, 0x00f2, 0x00f8, 0x00f4, 0x00ff, 0x00f0, 0xffff, 0x00e7, 0x00e0, 0x00ef, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x002f, 0x0020,
	0x0027, 0xffff, 0x0030, 0x003f, 0x002b, 0x0038, 0x002d, 0x002e, 0xffff, 0xffff, 0x0021,
	0x0022, 0x0033, 0x0024, 0x0035, 0x0036, 0x0037, 0x0028, 0x0039, 0x003a, 0x003b, 0x003c,
	0x003d, 0x003e, 0xffff, 0xffff, 0x003e, 0x003d, 0x0023, 0x003b, 0x0025, 0x0026, 0x0028,
	0x0037, 0x0029, 0x002a, 0x0024, 0x002c, 0x0022, 0x0021, 0xffff, 0xffff, 0x0031, 0x0032,
	0x0038, 0x0034, 0x003f, 0x0030, 0xffff, 0x0027, 0x0020, 0x002f, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x01bc, 0xffff, 0x004f, 0x0040, 0x0047, 0xffff,
	0x0050, 0x005f, 0x004b, 0x0058, 0x004d, 0x004e, 0xffff, 0xffff, 0x0041, 0x0042, 0x0053,
	0x0044, 0x0055, 0x0056, 0x0057, 0x0048, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e,
	0xffff, 0xffff, 0x005e, 0x005d, 0x0043, 0x005b, 0x0045, 0x0046, 0x0048, 0x0057, 0x0049,
	0x004a, 0x0044, 0x004c, 0x0042, 0x0041, 0xffff, 0xffff, 0x0051, 0x0052, 0x0058, 0x0054,
	0x005f, 0x0050, 0xffff, 0x0047, 0x0040, 0x004f, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x008f, 0x0080, 0x0087, 0xffff, 0x0090, 0x009f,
	0x008b, 0x0098, 0x008d, 0x008e, 0xffff, 0xffff, 0x0081, 0x0082, 0x0093, 0x0084, 0x0095,
	0x0096, 0x0097, 0x0088, 0x0099, 0x009a, 0x009b, 0x009c, 0x009d, 0x009e, 0xffff, 0xffff,
	0x009e, 0x009d, 0x0083, 0x009b, 0x0085, 0x0086, 0x0088, 0x0097, 0x0089, 0x008a, 0x0084,
	0x008c, 0x0082, 0x0081, 0xffff, 0xffff, 0x0091, 0x0092, 0x0098, 0x0094, 0x009f, 0x0090,
	0xffff, 0x0087, 0x0080, 0x008f, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0x006f, 0x0060, 0x0067, 0xffff, 0x0070, 0x007f, 0x006b, 0x0078,
	0x006d, 0x006e, 0xffff, 0xffff, 0x0061, 0x0062, 0x0073, 0x0064, 0x0075, 0x0076, 0x0077,
	0x0068, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0xffff, 0xffff, 0x007e, 0x007d,
	0x0063, 0x007b, 0x0065, 0x0066, 0x0068, 0x0077, 0x0069, 0x006a, 0x0064, 0x006c, 0x0062,
	0x0061, 0xffff, 0xffff, 0x0071, 0x0072, 0x0078, 0x0074, 0x007f, 0x0070, 0xffff, 0x0067,
	0x0060, 0x006f, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0x000f, 0x0000, 0x0007, 0xffff, 0x0010, 0x001f, 0x000b, 0x0018, 0x000d, 0x000e,
	0xffff, 0xffff, 0x0001, 0x0002, 0x0013, 0x0004, 0x0015, 0x0016, 0x0017, 0x0008, 0x0019,
	0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0xffff, 0xffff, 0x001e, 0x001d, 0x0003, 0x001b,
	0x0005, 0x0006, 0x0008, 0x0017, 0x0009, 0x000a, 0x0004, 0x000c, 0x0002, 0x0001, 0xffff,
	0xffff, 0x0011, 0x0012, 0x0018, 0x0014, 0x001f, 0x0010, 0xffff, 0x0007, 0x0000, 0x000f,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x00ef,
	0x00e0, 0x00e7, 0xffff, 0x00f0, 0x00ff, 0x00eb, 0x00f8, 0x00ed, 0x00ee, 0xffff, 0xffff,
	0x00e1, 0x00e2, 0x00f3, 0x00e4, 0x00f5, 0x00f6, 0x00f7, 0x00e8, 0x00f9, 0x00fa, 0x00fb,
	0x00fc, 0x00fd, 0x00fe, 0xffff, 0xffff, 0x00fe, 0x00fd, 0x00e3, 0x00fb, 0x00e5, 0x00e6,
	0x00e8, 0x00f7, 0x00e9, 0x00ea, 0x00e4, 0x00ec, 0x00e2, 0x00e1, 0xffff, 0xffff, 0x00f1,
	0x00f2, 0x00f8, 0x00f4, 0x00ff, 0x00f0, 0xffff, 0x00e7, 0x00e0, 0x00ef, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
	0xffff
};

#pragma GCC push_options
#pragma GCC optimize("O3")

static const u8 *seekMPEGTS(const u8 *in, size_t in_size)
{
	const u8 *p;
	for (p = in; p < in + in_size; p++) {
		if (*p != 0x47)
			continue;

		if (p + 188 * 2 >= in + in_size)
			break;

		if ((p[188] == 0x47) && (p[188 * 2] == 0x47))
			break;
	}

	return p;
}

size_t forward_v4l2_decode_asi(const u8 *in, u8 *out, size_t in_size, size_t out_size,
			       size_t *error_count)
{
	size_t i, cnt = 0;
	int err_cnt = 0, k_cnt = 0;
	const u8 *inp = in;
	u8 *outp = out;

	for (i = 0; (i < in_size) && (cnt < out_size); i += 5) {
		u64 word = (*((u32 *)inp)) | ((u64)(*(inp + 4)) << 32);
		int j;

		for (j = 0; j < 4; j++) {
			u16 dec = t10to8[(word >> (j * 10)) & 0x3FF];

			if ((dec & 0xFF00) == 0xFF00)
				err_cnt++;

			if (dec & 0x100) {
				k_cnt++;
				continue;
			}

			*outp = dec & 0xFF;
			outp++;
			cnt++;
		}
		inp = inp + 5;
	}

	if (error_count)
		*error_count = err_cnt;

	return (size_t)cnt;
}

size_t forward_v4l2_render_asi(const u8 *in, u8 *out, size_t in_size, size_t out_size)
{
	const u8 *inp = in, *outp = out;
	size_t i;
	int j;
	int rate = (u64)in_size * 540000 / (out_size / 10 * 8) + 1;
	int acc = 0, skips = 0;
	const u8 *hdrp = seekMPEGTS(in, in_size);
	int k_cnt = 0;

	if (rate > 540000 / 2)
		rate = rate * 190 / 188;
	else if (rate > 540000 / 3)
		rate = rate * 189 / 188;

	for (i = 0; i < out_size; i += 5) {
		u64 w = 0;
		for (j = 0; j < 4; j++) {
			if ((acc >= 540000) && (inp < in + in_size)) {
				u8 b = *inp;

				if (inp == hdrp) {
					if (skips < 2) {
						k_cnt++;
						skips++;
						continue;
					}
					hdrp += 188;
				}

				w |= ((u64)b | 0x100) << (j * 10);
				inp++;
				acc = (acc % 540000);
				skips = 0;
			} else {
				k_cnt++;
				skips++;
			}
			acc += rate;
		}
		*((u64 *)outp) = w;
		outp += 5;
	}

	return inp - in;
}

void forward_v4l2_clear_asi(u8 *out, size_t out_size)
{
	memset(out, 0, out_size);
}
