/*
   forward-si534x.c - si534x PLL driver for SoftLab-NSK Forward boards

   Copyright (C) 2017 - 2024 SoftLab-NSK <forward@softlab.tv>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "forward-pll.h"
#include "forward-si534x.h"
#include <linux/delay.h>

struct forward_si534x_mode {
	int in_freq_p;
	int in_freq_h;
	bool in_m;
	bool out_m;
	u64 Pn[3];
	u32 Pd[3];
	u64 Mn;
	u32 Md;
	u64 Bslow;
	u64 Bfast;
	u8 oof_div_sel[3];
	u32 oof_ratio_ref[3];
	u32 lol_clr_delay;
	u32 hold_15m_cyc_count;
	u16 pfd_en_delay;
};

struct forward_si534x {
	struct forward_pll pll;
	struct forward_si534x_mode current_mode;

	u8 bus;
	u8 bank;
};

const struct forward_si534x_mode forward_si534x_modes[] = {
	{ 13500000,
	  15625,
	  false,
	  false,
	  { 1, 864, 864 },
	  { 1, 1, 1 },
	  0x014E2000000000,
	  0x80000000,
	  0x03070F0F170F,
	  0x03070B0C2716,
	  { 0, 10, 10 },
	  { 0x12F684C, 0x1000000, 0x1000000 },
	  0x0003CB07,
	  0x01A0AB,
	  0x06AC },
	{ 13500000,
	  15734,
	  false,
	  false,
	  { 1, 858, 858 },
	  { 1, 1, 1 },
	  0x014BCE00000000,
	  0x80000000,
	  0x03070F0F170F,
	  0x03070B0C2716,
	  { 0, 10, 10 },
	  { 0x1318776, 0x1000000, 0x1000000 },
	  0x0003C997,
	  0x019DC6,
	  0x06A0 },
	// 720p24
	{ 74250000,
	  18000,
	  false,
	  false,
	  { 1, 4125, 4125 },
	  { 1, 1, 1 },
	  0x01220A00000000,
	  0x80000000,
	  0x07070F0F1810,
	  0x07070B0C2817,
	  { 0, 12, 12 },
	  { 0x15D867C, 0x1600000, 0x1600000 },
	  0x0003AFC1,
	  0x0169B1,
	  0x05C9 },
	// 720p23.97
	{ 74250000,
	  18000,
	  true,
	  false,
	  { 1, 4125, 4125 },
	  { 1, 1, 1 },
	  0x016AE950000000,
	  0xA0000000,
	  0x07070F0F1810,
	  0x07070B0C2817,
	  { 0, 12, 12 },
	  { 0x15D2D19, 0x15FA5FA, 0x15FA5FA },
	  0x0003AFEF,
	  0x016A0E,
	  0x05CB },
	{ 74250000,
	  18000,
	  true,
	  true,
	  { 1, 4125, 4125 },
	  { 1, 1, 1 },
	  0x016AE950000000,
	  0xA0000000,
	  0x07070F0F1810,
	  0x07070B0C2817,
	  { 0, 12, 12 },
	  { 0x15D2D19, 0x15FA5FA, 0x15FA5FA },
	  0x0000AF9A,
	  0x016A0E,
	  0x05CB },
	// 720p25
	{ 74250000,
	  18750,
	  false,
	  false,
	  { 1, 3960, 3960 },
	  { 1, 1, 1 },
	  0x01167000000000,
	  0x80000000,
	  0x07070F0F1810,
	  0x07070B0C2817,
	  { 0, 12, 12 },
	  { 0x16C16C1, 0x1600000, 0x1600000 },
	  0x0003A894,
	  0x015B39,
	  0x058F },
	// 720p50
	{ 74250000,
	  37500,
	  false,
	  false,
	  { 2, 3960, 3960 },
	  { 1, 1, 1 },
	  0x01167000000000,
	  0x80000000,
	  0x07070F0F1810,
	  0x07070B0C2817,
	  { 1, 12, 12 },
	  { 0x16C16C1, 0x1600000, 0x1600000 },
	  0x0003A894,
	  0x015B39,
	  0x058F },
	// 720p30
	{ 74250000,
	  22500,
	  false,
	  false,
	  { 1, 3300, 3300 },
	  { 1, 1, 1 },
	  0x00E80800000000,
	  0x80000000,
	  0x07070E0E1910,
	  0x07070B0C2716,
	  { 0, 12, 12 },
	  { 0x1B4E81B, 0x1600000, 0x1600000 },
	  0x00038BDF,
	  0x01215B,
	  0x04A1 },
	// 720p60
	{ 74250000,
	  45000,
	  false,
	  false,
	  { 2, 3300, 3300 },
	  { 1, 1, 1 },
	  0x00E80800000000,
	  0x80000000,
	  0x07070E0E1910,
	  0x07070B0C2716,
	  { 1, 12, 12 },
	  { 0x1B4E81B, 0x1600000, 0x1600000 },
	  0x00038BDF,
	  0x01215B,
	  0x04A1 },
	// 720p29.97
	{ 74250000,
	  22500,
	  true,
	  false,
	  { 1, 3300, 3300 },
	  { 1, 1, 1 },
	  0x01225440000000,
	  0xA0000000,
	  0x07070E0E1910,
	  0x07070B0C2716,
	  { 0, 12, 12 },
	  { 0x1B4785F, 0x15FA5FA, 0x15FA5FA },
	  0x00038C04,
	  0x0121A5,
	  0x04A2 },
	{ 74250000,
	  22500,
	  true,
	  true,
	  { 1, 3300, 3300 },
	  { 1, 1, 1 },
	  0x01225440000000,
	  0xA0000000,
	  0x07070E0E1910,
	  0x07070B0C2716,
	  { 0, 12, 12 },
	  { 0x1B4785F, 0x15FA5FA, 0x15FA5FA },
	  0x00038C04,
	  0x0121A5,
	  0x04A2 },
	// 720p59.94
	{ 74250000,
	  22500,
	  true,
	  false,
	  { 2, 3300, 3300 },
	  { 1, 1, 1 },
	  0x01225440000000,
	  0xA0000000,
	  0x07070E0E1910,
	  0x07070B0C2716,
	  { 1, 12, 12 },
	  { 0x1B4785F, 0x15FA5FA, 0x15FA5FA },
	  0x00038C04,
	  0x0121A5,
	  0x04A2 },
	{ 74250000,
	  22500,
	  true,
	  true,
	  { 2, 3300, 3300 },
	  { 1, 1, 1 },
	  0x01225440000000,
	  0xA0000000,
	  0x07070E0E1910,
	  0x07070B0C2716,
	  { 1, 12, 12 },
	  { 0x1B4785F, 0x15FA5FA, 0x15FA5FA },
	  0x00038C04,
	  0x0121A5,
	  0x04A2 },
	// 1080p24
	{ 74250000,
	  27000,
	  false,
	  false,
	  { 1, 2750, 2750 },
	  { 1, 1, 1 },
	  0x00C15C00000000,
	  0x80000000,
	  0x07070F0F170F,
	  0x07070B0C2716,
	  { 1, 12, 12 },
	  { 0x10624DD, 0x1600000, 0x1600000 },
	  0x0003EB94,
	  0x00F121,
	  0x03DB },
	// 1080p23.98
	{ 74250000,
	  27000,
	  true,
	  false,
	  { 1, 2750, 2750 },
	  { 1, 1, 1 },
	  0x00C18D80000000,
	  0x80000000,
	  0x07070F0F170F,
	  0x07070B0C2716,
	  { 1, 12, 12 },
	  { 0x105E1D2, 0x15FA5FA, 0x15FA5FA },
	  0x0003EBD1,
	  0x00F15F,
	  0x03DC },
	{ 74250000,
	  27000,
	  true,
	  true,
	  { 1, 2750, 2750 },
	  { 1, 1, 1 },
	  0x00C18D80000000,
	  0x80000000,
	  0x07070F0F170F,
	  0x07070B0C2716,
	  { 1, 12, 12 },
	  { 0x105E1D2, 0x15FA5FA, 0x15FA5FA },
	  0x0003EBD1,
	  0x00F15F,
	  0x03DC },
	// 1080p25/i50
	{ 74250000,
	  28125,
	  false,
	  false,
	  { 1, 2640, 2640 },
	  { 1, 1, 1 },
	  0x00B9A000000000,
	  0x80000000,
	  0x07070F0F170F,
	  0x07070B0C2716,
	  { 1, 12, 12 },
	  { 0x1111111, 0x1600000, 0x1600000 },
	  0x0003E202,
	  0x00E77C,
	  0x03B3 },
	// 1080p50
	{ 148500000,
	  56250,
	  false,
	  false,
	  { 2, 5280, 5280 },
	  { 1, 1, 1 },
	  0x00B9A000000000,
	  0x80000000,
	  0x07070F0F170F,
	  0x07070B0C2716,
	  { 2, 13, 13 },
	  { 0x1111111, 0x1600000, 0x1600000 },
	  0x0003E202,
	  0x00E77C,
	  0x03B3 },
	// 1080p30/i60
	{ 74250000,
	  33750,
	  false,
	  false,
	  { 1, 2200, 2200 },
	  { 1, 1, 1 },
	  0x009AB000000000,
	  0x80000000,
	  0x07030F0F1810,
	  0x07030B0C2817,
	  { 1, 12, 12 },
	  { 0x147AE14, 0x1600000, 0x1600000 },
	  0x0003BBBA,
	  0x00C0E7,
	  0x0315 },
	// 1080p60
	{ 148500000,
	  67500,
	  false,
	  false,
	  { 2, 4400, 4400 },
	  { 1, 1, 1 },
	  0x009AB000000000,
	  0x80000000,
	  0x07030F0F1810,
	  0x07030B0C2817,
	  { 2, 13, 13 },
	  { 0x147AE14, 0x1600000, 0x1600000 },
	  0x0003BBBA,
	  0x00C0E7,
	  0x0315 },
	// 1080p29.97/i59.94
	{ 74250000,
	  33750,
	  true,
	  false,
	  { 1, 2200, 2200 },
	  { 1, 1, 1 },
	  0x00C18D80000000,
	  0xA0000000,
	  0x07030F0F1810,
	  0x07030B0C2817,
	  { 1, 12, 12 },
	  { 0x1475A47, 0x15FA5FA, 0x15FA5FA },
	  0x0003BBEB,
	  0x00C119,
	  0x0316 },
	{ 74250000,
	  33750,
	  true,
	  true,
	  { 1, 2200, 2200 },
	  { 1, 1, 1 },
	  0x00C18D80000000,
	  0xA0000000,
	  0x07030F0F1810,
	  0x07030B0C2817,
	  { 1, 12, 12 },
	  { 0x1475A47, 0x15FA5FA, 0x15FA5FA },
	  0x0003BBEB,
	  0x00C119,
	  0x0316 },
	// 1080p59.94
	{ 148500000,
	  67500,
	  true,
	  false,
	  { 2, 4400, 4400 },
	  { 1, 1, 1 },
	  0x00C18D80000000,
	  0xA0000000,
	  0x07030F0F1810,
	  0x07030B0C2817,
	  { 2, 13, 13 },
	  { 0x1475A47, 0x15FA5FA, 0x15FA5FA },
	  0x0003BBEB,
	  0x00C119,
	  0x0316 },
	{ 148500000,
	  67500,
	  true,
	  true,
	  { 2, 4400, 4400 },
	  { 1, 1, 1 },
	  0x00C18D80000000,
	  0xA0000000,
	  0x07030F0F1810,
	  0x07030B0C2817,
	  { 2, 13, 13 },
	  { 0x1475A47, 0x15FA5FA, 0x15FA5FA },
	  0x0003BBEB,
	  0x00C119,
	  0x0316 },
};

const struct forward_si534x_mode si534x_default_mode = { -1,
							 -1,
							 false,
							 false,
							 { 1, 2640, 2640 },
							 { 1, 1, 1 },
							 0x00B9A000000000,
							 0x80000000,
							 0x07070F0F170F,
							 0x0707090A2B18,
							 { 1, 12, 12 },
							 { 0x1111111, 0x1600000, 0x1600000 },
							 0x0000E1AD,
							 0x00E77C,
							 0x03B3 };
const struct forward_si534x_mode si534x_default_mode_m = { -1,
							   -1,
							   true,
							   false,
							   { 1, 2200, 2200 },
							   { 1, 1, 1 },
							   0x00C18D80000000,
							   0xA0000000,
							   0x07030F0F1810,
							   0x070308092E1A,
							   { 1, 12, 12 },
							   { 0x1475A47, 0x15FA5FA, 0x15FA5FA },
							   0x0000BB96,
							   0x00C119,
							   0x0316 };

const struct forward_si534x_mode si534x_default_mode_m_m = { -1,
							     -1,
							     true,
							     true,
							     { 1, 2200, 2200 },
							     { 1, 1, 1 },
							     0x00C18D80000000,
							     0xA0000000,
							     0x07030F0F1810,
							     0x070308092E1A,
							     { 1, 12, 12 },
							     { 0x1475A47, 0x15FA5FA, 0x15FA5FA },
							     0x0000BB96,
							     0x00C119,
							     0x0316 };
const int forward_si534x_modes_count =
	sizeof(forward_si534x_modes) / sizeof(struct forward_si534x_mode);

inline void si534x_select_bank(struct forward_si534x *si534x, u16 addr)
{
	u8 bank = ((addr >> 8) & 0xFF);
	u8 tx[4] = { 0x00, 0x01, 0x40, bank };

	if (si534x->bank != bank)
		si534x->pll.dev->spi_xfer(si534x->pll.dev, si534x->bus, tx, 32, NULL, 0);
	si534x->bank = bank;
}

inline void si534x_write_data(struct forward_si534x *si534x, u16 addr, const u8 *data, u8 count)
{
	u8 tx[2 + 16];
	int i;

	si534x_select_bank(si534x, addr);

	tx[0] = 0x00;
	tx[1] = (addr & 0xFF);
	for (i = 0; i < count; i++) {
		tx[2 + i * 2] = 0x60;
		tx[2 + i * 2 + 1] = data[i];
	}

	si534x->pll.dev->spi_xfer(si534x->pll.dev, si534x->bus, tx, (2 + count * 2) * 8, NULL, 0);
}

inline void si534x_read_data(struct forward_si534x *si534x, u16 addr, u8 *data, u8 count)
{
	u8 tx[2 + 16];
	u8 rx[2 + 16];
	int i;

	si534x_select_bank(si534x, addr);

	tx[0] = 0x00;
	tx[1] = (addr & 0xFF);
	for (i = 0; i < count; i++) {
		tx[2 + i * 2] = 0xA0;
		tx[2 + i * 2 + 1] = 0x00;
	}
	si534x->pll.dev->spi_xfer(si534x->pll.dev, si534x->bus, tx, (2 + count * 2) * 8, rx,
				  (2 + count * 2) * 8);
	for (i = 0; i < count; i++)
		data[i] = rx[2 + i * 2 + 1];
}

static void si534x_write_reg_8(struct forward_si534x *si534x, u16 addr, u8 data)
{
	si534x_write_data(si534x, addr, &data, 1);
}

//static void si534x_read_reg_8(struct forward_si534x *si534x, u16 addr, u8 *data)
//{
//	si534x_read_data(si534x, addr, data, 1);
//}

static void si534x_write_reg_16(struct forward_si534x *si534x, u16 addr, u16 data)
{
	si534x_write_data(si534x, addr, (u8 *)&data, 2);
}

//static void si534x_read_reg_16(struct forward_si534x *si534x, u16 addr, u16 *data)
//{
//	si534x_read_data(si534x, addr, (u8*)data, 2);
//}

static void si534x_write_reg_24(struct forward_si534x *si534x, u16 addr, u32 data)
{
	si534x_write_data(si534x, addr, (u8 *)&data, 3);
}

//static void si534x_read_reg_24(struct forward_si534x *si534x, u16 addr, u32 *data)
//{
//	si534x_read_data(si534x, addr, (u8*)data, 3);
//}

static void si534x_write_reg_32(struct forward_si534x *si534x, u16 addr, u32 data)
{
	si534x_write_data(si534x, addr, (u8 *)&data, 4);
}

static void si534x_read_reg_32(struct forward_si534x *si534x, u16 addr, u32 *data)
{
	si534x_read_data(si534x, addr, (u8 *)data, 4);
}

static void si534x_write_reg_48(struct forward_si534x *si534x, u16 addr, u64 data)
{
	si534x_write_data(si534x, addr, (u8 *)&data, 6);
}

//static void si534x_read_reg_48(struct forward_si534x *si534x, u16 addr, u64 *data)
//{
//	si534x_read_data(si534x, addr, (u8*)data, 6);
//}

static void si534x_write_reg_56(struct forward_si534x *si534x, u16 addr, u64 data)
{
	si534x_write_data(si534x, addr, (u8 *)&data, 7);
}

//static void si534x_read_reg_56(struct forward_si534x *si534x, u16 addr, u64 *data)
//{
//	si534x_read_data(si534x, addr, (u8*)data, 7);
//}

static bool si534x_need_mode_switch(struct forward_pll *pll, struct forward_pll_mode *mode)
{
	struct forward_si534x *si534x = container_of(pll, struct forward_si534x, pll);
	struct forward_si534x_mode *m1 = (struct forward_si534x_mode *)mode;
	struct forward_si534x_mode *m2 = &si534x->current_mode;

	return (m1->in_freq_p != m2->in_freq_p) || (m1->in_freq_h != m2->in_freq_h) ||
	       (m1->Pn[0] != m2->Pn[0]) || (m1->Pn[1] != m2->Pn[1]) || (m1->Pn[2] != m2->Pn[2]) ||
	       (m1->Pd[0] != m2->Pd[0]) || (m1->Pd[1] != m2->Pd[1]) || (m1->Pd[2] != m2->Pd[2]) ||
	       (m1->Mn != m2->Mn) || (m1->Md != m2->Md);
}

static struct forward_pll_mode *si534x_find_mode(struct forward_pll *pll, s64 in_pix_freq,
						 s64 in_h_freq, bool in_m, s64 out_freq, bool out_m)
{
	int i;

	if (out_freq != 148500000)
		return NULL;

	if ((in_pix_freq <= 0) && (in_h_freq <= 0)) {
		if (!in_m)
			return (struct forward_pll_mode *)&si534x_default_mode;
		else if (!out_m)
			return (struct forward_pll_mode *)&si534x_default_mode_m;
		else
			return (struct forward_pll_mode *)&si534x_default_mode_m_m;
	}

	for (i = 0; i < forward_si534x_modes_count; i++) {
		const struct forward_si534x_mode *cur = &forward_si534x_modes[i];
		if (((in_pix_freq <= 0) || (cur->in_freq_p == in_pix_freq)) &&
		    ((in_h_freq <= 0) || (cur->in_freq_h == in_h_freq)) && (cur->in_m == in_m) &&
		    (cur->out_m == out_m)) {
			return (struct forward_pll_mode *)cur;
		}
	}

	return NULL;
}

static int si534x_switch_mode(struct forward_pll *pll, struct forward_pll_mode *mode)
{
	struct forward_si534x *si534x = container_of(pll, struct forward_si534x, pll);
	struct forward_si534x_mode *m = (struct forward_si534x_mode *)mode;

	si534x_write_reg_8(si534x, 0x0B24, 0xC0);
	si534x_write_reg_8(si534x, 0x0B25, 0x00);
	si534x_write_reg_8(si534x, 0x0540, 0x01);

	msleep(300);

	si534x_write_reg_8(si534x, 0x0041, m->oof_div_sel[0]);
	si534x_write_reg_8(si534x, 0x0042, m->oof_div_sel[1]);
	si534x_write_reg_8(si534x, 0x0043, m->oof_div_sel[2]);

	si534x_write_reg_32(si534x, 0x005A, m->oof_ratio_ref[0]);
	si534x_write_reg_32(si534x, 0x005E, m->oof_ratio_ref[1]);
	si534x_write_reg_32(si534x, 0x0062, m->oof_ratio_ref[2]);
	si534x_write_reg_32(si534x, 0x00A9, m->lol_clr_delay);

	si534x_write_reg_48(si534x, 0x0208, m->Pn[0]);
	si534x_write_reg_32(si534x, 0x020E, m->Pd[0]);
	si534x_write_reg_48(si534x, 0x0212, m->Pn[1]);
	si534x_write_reg_32(si534x, 0x0218, m->Pd[1]);
	si534x_write_reg_48(si534x, 0x021C, m->Pn[2]);
	si534x_write_reg_32(si534x, 0x0222, m->Pd[2]);
	si534x_write_reg_8(si534x, 0x0230, 0x07);

	si534x_write_reg_48(si534x, 0x0508, m->Bslow);
	si534x_write_reg_48(si534x, 0x050E, m->Bfast);
	si534x_write_reg_8(si534x, 0x0514, 0x01);

	si534x_write_reg_56(si534x, 0x0515, m->Mn);
	si534x_write_reg_32(si534x, 0x051C, m->Md);
	si534x_write_reg_8(si534x, 0x0520, 0x1);

	si534x_write_reg_24(si534x, 0x0532, m->hold_15m_cyc_count);
	si534x_write_reg_16(si534x, 0x0589, m->pfd_en_delay);

	//si534x_write_reg_24(si534x, 0x0250, 0x00);
	//si534x_write_reg_24(si534x, 0x0253, 0x00);

	if (m->out_m) {
		si534x_write_reg_48(si534x, 0x0302, 0x2331000000ULL);
		si534x_write_reg_32(si534x, 0x0308, 0xc8000000);
		si534x_write_reg_8(si534x, 0x030c, 0x1);
	} else {
		si534x_write_reg_48(si534x, 0x0302, 0x2328000000ULL);
		si534x_write_reg_32(si534x, 0x0308, 0xc8000000);
		si534x_write_reg_8(si534x, 0x030c, 0x1);
	}
	si534x_write_reg_48(si534x, 0x030d, 0x2331000000ULL);
	si534x_write_reg_32(si534x, 0x0313, 0xc8000000);
	si534x_write_reg_8(si534x, 0x0317, 0x1);

	si534x_write_reg_8(si534x, 0x001C, 0x01);
	si534x_write_reg_8(si534x, 0x0540, 0x00);
	si534x_write_reg_8(si534x, 0x0B24, 0xC3);
	si534x_write_reg_8(si534x, 0x0B25, 0x02);

	si534x->current_mode = *m;

	return 0;
}

static int si534x_select_input(struct forward_pll *pll, int input, bool master, bool autoswitch)
{
	struct forward_si534x *si534x = container_of(pll, struct forward_si534x, pll);

	si534x_write_reg_8(si534x, 0x052A, (input << 1) | 0x1);

	if (autoswitch) {
		switch (input) {
		case 0:
			si534x_write_reg_16(si534x, 0x0538, 0x0321);
			break;
		case 1:
			si534x_write_reg_16(si534x, 0x0538, 0x0213);
			break;
		case 2:
			si534x_write_reg_16(si534x, 0x0538, 0x0123);
			break;
		}
		si534x_write_reg_8(si534x, 0x0536, 0x6);
	} else
		si534x_write_reg_8(si534x, 0x0536, 0x4);

	si534x_write_reg_8(si534x, 0x0535, master ? 1 : 0);

	return 0;
}

static s64 si534x_tune(struct forward_pll *pll, s64 ppt)
{
	struct forward_si534x *si534x = container_of(pll, struct forward_si534x, pll);
	u32 denom = 0xc8000000 + 0xc8000000LL * ppt / 1000000000000LL;

	si534x_write_reg_32(si534x, 0x308, denom);
	si534x_write_reg_32(si534x, 0x313, denom);
	si534x_write_reg_32(si534x, 0x338, 2);

	return ppt;
}

static s64 si534x_get_tune(struct forward_pll *pll)
{
	struct forward_si534x *si534x = container_of(pll, struct forward_si534x, pll);
	u32 denom;

	si534x_read_reg_32(si534x, 0x308, &denom);

	return (s32)(denom - 0xc8000000U) * 1000000000000LL / 0xc8000000LL;
}

static void si534x_calibrate(struct forward_pll *pll)
{
	struct forward_si534x *si534x = container_of(pll, struct forward_si534x, pll);
	si534x_write_reg_8(si534x, 0x001C, 0x4);
}

static struct forward_pll_ops si534x_ops = {
	.find_mode = si534x_find_mode,
	.need_mode_switch = si534x_need_mode_switch,
	.switch_mode = si534x_switch_mode,
	.select_input = si534x_select_input,
	.tune = si534x_tune,
	.get_tune = si534x_get_tune,
	.calibrate = si534x_calibrate,
};

int forward_si534x_init(struct forward_dev *dev, struct forward_pll **pll, u8 bus, int num_inputs)
{
	struct forward_si534x *si534x = kzalloc(sizeof(struct forward_si534x), GFP_KERNEL);

	if (!si534x)
		return -ENOMEM;

	si534x->pll.dev = dev;
	si534x->pll.features = FORWARD_PLL_FEATURE_FALLBACK | FORWARD_PLL_FEATURE_MASTER |
			       FORWARD_PLL_FEATURE_TUNE;
	si534x->pll.num_inputs = num_inputs;
	si534x->pll.ops = &si534x_ops;
	si534x->bus = bus;
	si534x->bank = 0xFF;
	si534x->current_mode = si534x_default_mode;

	*pll = &si534x->pll;

	return 0;
}
EXPORT_SYMBOL(forward_si534x_init);

void forward_si534x_fini(struct forward_dev *dev, struct forward_pll **pll)
{
	if (pll) {
		struct forward_si534x *si534x = container_of(*pll, struct forward_si534x, pll);
		kfree(si534x);
		*pll = NULL;
	}
}
EXPORT_SYMBOL(forward_si534x_fini);
