/*
   fd720_reg.h - register definitions driver for SoftLab-NSK FD720 board

   Copyright (C) 2017 - 2024 SoftLab-NSK <forward@softlab.tv>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef FD720_REG_H
#define FD720_REG_H

#include <linux/types.h>
#include <linux/io.h>

#define __read_reg(csr, addr) ioread32(&csr[addr])
#define __write_reg(csr, addr, v) iowrite32(v, &csr[addr])

typedef struct _FD720_VDMA {
	uint32_t _unused3000 : 31;
	uint32_t enable : 1;
} __attribute__((packed)) FD720_VDMA;

#define FD720_VDMA_A (0x00)

__always_inline FD720_VDMA FD720_VDMA_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_VDMA_A);
	return *((FD720_VDMA *)&r);
}

__always_inline void FD720_VDMA_W(uint32_t *csr, FD720_VDMA v)
{
	__write_reg(csr, FD720_VDMA_A, *((uint32_t *)&v));
}

typedef struct _FD720_VDMADescriptor {
	uint32_t address : 32;
} __attribute__((packed)) FD720_VDMADescriptor;

#define FD720_VDMADescriptor_A (0x01)

__always_inline FD720_VDMADescriptor FD720_VDMADescriptor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_VDMADescriptor_A);
	return *((FD720_VDMADescriptor *)&r);
}

__always_inline void FD720_VDMADescriptor_W(uint32_t *csr, FD720_VDMADescriptor v)
{
	__write_reg(csr, FD720_VDMADescriptor_A, *((uint32_t *)&v));
}

typedef struct _FD720_IRQEnable {
	uint32_t in0IRQ : 1;
	uint32_t _unused0201 : 2;
	uint32_t in1IRQ : 1;
	uint32_t _unused0504 : 2;
	uint32_t sync0IRQ : 1;
	uint32_t _unused0707 : 1;
	uint32_t sync1IRQ : 1;
	uint32_t _unused3109 : 23;
} __attribute__((packed)) FD720_IRQEnable;

#define FD720_IRQEnable_A (0x04)

__always_inline FD720_IRQEnable FD720_IRQEnable_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_IRQEnable_A);
	return *((FD720_IRQEnable *)&r);
}

__always_inline void FD720_IRQEnable_W(uint32_t *csr, FD720_IRQEnable v)
{
	__write_reg(csr, FD720_IRQEnable_A, *((uint32_t *)&v));
}

typedef struct _FD720_IRQFlags {
	uint32_t in0IRQ : 1;
	uint32_t in0Buffer : 2;
	uint32_t in1IRQ : 1;
	uint32_t in1Buffer : 2;
	uint32_t sync0IRQ : 1;
	uint32_t sync0Field : 1;
	uint32_t sync1IRQ : 1;
	uint32_t sync1Field : 1;
	uint32_t _unused3110 : 22;
} __attribute__((packed)) FD720_IRQFlags;

#define FD720_IRQFlags_A (0x05)

__always_inline FD720_IRQFlags FD720_IRQFlags_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_IRQFlags_A);
	return *((FD720_IRQFlags *)&r);
}

__always_inline void FD720_IRQFlags_W(uint32_t *csr, FD720_IRQFlags v)
{
	__write_reg(csr, FD720_IRQFlags_A, *((uint32_t *)&v));
}

typedef struct _FD720_VideoInCS {
	uint32_t enable : 1;
	uint32_t hpd : 1;
	uint32_t rateLocked : 1;
	uint32_t rate : 3;
	uint32_t formatLocked : 1;
	uint32_t progressive : 1;
	uint32_t capture : 1;
	uint32_t captureColor : 2;
	uint32_t captureDepth : 2;
	uint32_t pixrep : 1;
	uint32_t dvi : 1;
	uint32_t edidOverride : 1;
	uint32_t bitrate : 15;
	uint32_t test : 1;
} __attribute__((packed)) FD720_VideoInCS;

#define FD720_VideoInCS_A(n) (0x40 + n * 16)

__always_inline FD720_VideoInCS FD720_VideoInCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD720_VideoInCS_A(n));
	return *((FD720_VideoInCS *)&r);
}

__always_inline void FD720_VideoInCS_W(uint32_t *csr, unsigned int n, FD720_VideoInCS v)
{
	__write_reg(csr, FD720_VideoInCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD720_VideoInLine {
	uint32_t totalLines : 12;
	uint32_t _unused1512 : 4;
	uint32_t activeLines : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD720_VideoInLine;

#define FD720_VideoInLine_A(n) (0x41 + n * 16)

__always_inline FD720_VideoInLine FD720_VideoInLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD720_VideoInLine_A(n));
	return *((FD720_VideoInLine *)&r);
}

typedef struct _FD720_VideoInPixel {
	uint32_t totalPixels : 14;
	uint32_t _unused1514 : 2;
	uint32_t activePixels : 14;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD720_VideoInPixel;

#define FD720_VideoInPixel_A(n) (0x42 + n * 16)

__always_inline FD720_VideoInPixel FD720_VideoInPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD720_VideoInPixel_A(n));
	return *((FD720_VideoInPixel *)&r);
}

typedef struct _FD720_VideoInIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD720_VideoInIRQRefTime;

#define FD720_VideoInIRQRefTime_A(n) (0x44 + n * 16)

__always_inline FD720_VideoInIRQRefTime FD720_VideoInIRQRefTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD720_VideoInIRQRefTime_A(n));
	return *((FD720_VideoInIRQRefTime *)&r);
}

typedef struct _FD720_VideoInANCCounter {
	uint32_t counter : 21;
	uint32_t _unused3121 : 11;
} __attribute__((packed)) FD720_VideoInANCCounter;

#define FD720_VideoInANCCounter_A(n) (0x46 + n * 16)

__always_inline FD720_VideoInANCCounter FD720_VideoInANCCounter_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD720_VideoInANCCounter_A(n));
	return *((FD720_VideoInANCCounter *)&r);
}

typedef struct _FD720_VideoInInfoFrame {
	uint32_t aviVIC : 8;
	uint32_t aviY : 3;
	uint32_t aviYQ : 2;
	uint32_t aviPR : 4;
	uint32_t aviValid : 1;
	uint32_t aiCT : 4;
	uint32_t aiCC : 3;
	uint32_t aiSF : 3;
	uint32_t aiSS : 2;
	uint32_t aiValid : 1;
	uint32_t _unused3131 : 1;
} __attribute__((packed)) FD720_VideoInInfoFrame;

#define FD720_VideoInInfoFrame_A(n) (0x47 + n * 16)

__always_inline FD720_VideoInInfoFrame FD720_VideoInInfoFrame_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD720_VideoInInfoFrame_A(n));
	return *((FD720_VideoInInfoFrame *)&r);
}

typedef struct _FD720_VideoInFIFOStatus {
	uint32_t videoMax : 15;
	uint32_t videoError : 1;
	uint32_t audioMax : 15;
	uint32_t audioError : 1;
} __attribute__((packed)) FD720_VideoInFIFOStatus;

#define FD720_VideoInFIFOStatus_A(n) (0x4A + n * 16)

__always_inline FD720_VideoInFIFOStatus FD720_VideoInFIFOStatus_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD720_VideoInFIFOStatus_A(n));
	return *((FD720_VideoInFIFOStatus *)&r);
}

typedef struct _FD720_VideoInColorConversion {
	uint32_t data : 16;
	uint32_t address : 4;
	uint32_t write : 1;
	uint32_t clamp : 1;
	uint32_t clampRGB : 1;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD720_VideoInColorConversion;

#define FD720_VideoInColorConversion_A(n) (0x4C + n * 16)

__always_inline FD720_VideoInColorConversion FD720_VideoInColorConversion_R(uint32_t *csr,
									    unsigned int n)
{
	uint32_t r = __read_reg(csr, FD720_VideoInColorConversion_A(n));
	return *((FD720_VideoInColorConversion *)&r);
}

__always_inline void FD720_VideoInColorConversion_W(uint32_t *csr, unsigned int n,
						    FD720_VideoInColorConversion v)
{
	__write_reg(csr, FD720_VideoInColorConversion_A(n), *((uint32_t *)&v));
}

typedef struct _FD720_GlobalCSR {
	uint32_t videoReset : 1;
	uint32_t icapRW : 1;
	uint32_t reboot : 1;
	uint32_t _unused3103 : 29;
} __attribute__((packed)) FD720_GlobalCSR;

#define FD720_GlobalCSR_A (0x80)

__always_inline FD720_GlobalCSR FD720_GlobalCSR_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_GlobalCSR_A);
	return *((FD720_GlobalCSR *)&r);
}

__always_inline void FD720_GlobalCSR_W(uint32_t *csr, FD720_GlobalCSR v)
{
	__write_reg(csr, FD720_GlobalCSR_A, *((uint32_t *)&v));
}

typedef struct _FD720_TemperatureMonitor {
	uint32_t temperature : 10;
	uint32_t maxTemperature : 10;
	uint32_t minTemperature : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD720_TemperatureMonitor;

#define FD720_TemperatureMonitor_A (0x82)

__always_inline FD720_TemperatureMonitor FD720_TemperatureMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_TemperatureMonitor_A);
	return *((FD720_TemperatureMonitor *)&r);
}

typedef struct _FD720_VCoreMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD720_VCoreMonitor;

#define FD720_VCoreMonitor_A (0x83)

__always_inline FD720_VCoreMonitor FD720_VCoreMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_VCoreMonitor_A);
	return *((FD720_VCoreMonitor *)&r);
}

typedef struct _FD720_VAuxMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD720_VAuxMonitor;

#define FD720_VAuxMonitor_A (0x84)

__always_inline FD720_VAuxMonitor FD720_VAuxMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_VAuxMonitor_A);
	return *((FD720_VAuxMonitor *)&r);
}

typedef struct _FD720_VideoClockCounter {
	uint32_t counter : 32;
} __attribute__((packed)) FD720_VideoClockCounter;

#define FD720_VideoClockCounter_A (0x89)

__always_inline FD720_VideoClockCounter FD720_VideoClockCounter_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_VideoClockCounter_A);
	return *((FD720_VideoClockCounter *)&r);
}

typedef struct _FD720_SoftID {
	uint32_t id : 32;
} __attribute__((packed)) FD720_SoftID;

#define FD720_SoftID_A (0x8F)

__always_inline FD720_SoftID FD720_SoftID_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_SoftID_A);
	return *((FD720_SoftID *)&r);
}

typedef struct _FD720_PCIEPerfomance {
	uint32_t txCounter : 16;
	uint32_t rxCounter : 16;
} __attribute__((packed)) FD720_PCIEPerfomance;

#define FD720_PCIEPerfomance_A (0x92)

__always_inline FD720_PCIEPerfomance FD720_PCIEPerfomance_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_PCIEPerfomance_A);
	return *((FD720_PCIEPerfomance *)&r);
}

typedef struct _FD720_PCIERxMaxLatency {
	uint32_t maxLatency : 16;
	uint32_t numRequests : 16;
} __attribute__((packed)) FD720_PCIERxMaxLatency;

#define FD720_PCIERxMaxLatency_A (0x93)

__always_inline FD720_PCIERxMaxLatency FD720_PCIERxMaxLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_PCIERxMaxLatency_A);
	return *((FD720_PCIERxMaxLatency *)&r);
}

typedef struct _FD720_PCIERxTotalLatency {
	uint32_t latency : 32;
} __attribute__((packed)) FD720_PCIERxTotalLatency;

#define FD720_PCIERxTotalLatency_A (0x94)

__always_inline FD720_PCIERxTotalLatency FD720_PCIERxTotalLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_PCIERxTotalLatency_A);
	return *((FD720_PCIERxTotalLatency *)&r);
}

typedef struct _FD720_StaticVersion {
	uint32_t revision : 16;
	uint32_t major : 8;
	uint32_t hardware : 4;
	uint32_t project : 4;
} __attribute__((packed)) FD720_StaticVersion;

#define FD720_StaticVersion_A (0xFF)

__always_inline FD720_StaticVersion FD720_StaticVersion_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD720_StaticVersion_A);
	return *((FD720_StaticVersion *)&r);
}

#endif
