/*
   fd722_reg.h - register definitions driver for SoftLab-NSK FD722 board

   Copyright (C) 2017 - 2024 SoftLab-NSK <forward@softlab.tv>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef FD722_REG_H
#define FD722_REG_H

#include <linux/types.h>
#include <linux/io.h>

#define __read_reg(csr, addr) ioread32(&csr[addr])
#define __write_reg(csr, addr, v) iowrite32(v, &csr[addr])

typedef struct _FD722_GlobalCSR {
	uint32_t _unused0000 : 1;
	uint32_t videoReset : 1;
	uint32_t vpllReset : 1;
	uint32_t icapRW : 1;
	uint32_t _unused0404 : 1;
	uint32_t reconfigure : 1;
	uint32_t vpllClkSelect : 1;
	uint32_t reboot : 1;
	uint32_t _unused1708 : 10;
	uint32_t vpllLocked : 1;
	uint32_t qpllLocked : 1;
	uint32_t vpllClk1Bad : 1;
	uint32_t vpllClk2Bad : 1;
	uint32_t _unused3122 : 10;
} __attribute__((packed)) FD722_GlobalCSR;

#define FD722_GlobalCSR_A (0x80)

__always_inline FD722_GlobalCSR FD722_GlobalCSR_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_GlobalCSR_A);
	return *((FD722_GlobalCSR *)&r);
}

__always_inline void FD722_GlobalCSR_W(uint32_t *csr, FD722_GlobalCSR v)
{
	__write_reg(csr, FD722_GlobalCSR_A, *((uint32_t *)&v));
}

typedef struct _FD722_TemperatureMonitor {
	uint32_t temperature : 10;
	uint32_t maxTemperature : 10;
	uint32_t minTemperature : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD722_TemperatureMonitor;

#define FD722_TemperatureMonitor_A (0x82)

__always_inline FD722_TemperatureMonitor FD722_TemperatureMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_TemperatureMonitor_A);
	return *((FD722_TemperatureMonitor *)&r);
}

typedef struct _FD722_VCoreMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD722_VCoreMonitor;

#define FD722_VCoreMonitor_A (0x83)

__always_inline FD722_VCoreMonitor FD722_VCoreMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_VCoreMonitor_A);
	return *((FD722_VCoreMonitor *)&r);
}

typedef struct _FD722_VAuxMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD722_VAuxMonitor;

#define FD722_VAuxMonitor_A (0x84)

__always_inline FD722_VAuxMonitor FD722_VAuxMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_VAuxMonitor_A);
	return *((FD722_VAuxMonitor *)&r);
}

typedef struct _FD722_SPI {
	uint32_t data : 16;
	uint32_t _unused2316 : 8;
	uint32_t cs : 1;
	uint32_t bit8 : 1;
	uint32_t start : 1;
	uint32_t bus : 1;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD722_SPI;

#define FD722_SPI_A (0x86)

__always_inline FD722_SPI FD722_SPI_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_SPI_A);
	return *((FD722_SPI *)&r);
}

__always_inline void FD722_SPI_W(uint32_t *csr, FD722_SPI v)
{
	__write_reg(csr, FD722_SPI_A, *((uint32_t *)&v));
}

typedef struct _FD722_I2C {
	uint32_t data : 8;
	uint32_t subAddress : 8;
	uint32_t rw : 1;
	uint32_t address : 7;
	uint32_t cs : 1;
	uint32_t nack : 1;
	uint32_t _unused3126 : 6;
} __attribute__((packed)) FD722_I2C;

#define FD722_I2C_A (0x87)

__always_inline FD722_I2C FD722_I2C_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_I2C_A);
	return *((FD722_I2C *)&r);
}

__always_inline void FD722_I2C_W(uint32_t *csr, FD722_I2C v)
{
	__write_reg(csr, FD722_I2C_A, *((uint32_t *)&v));
}

typedef struct _FD722_VideoClockCounter {
	uint32_t counter : 32;
} __attribute__((packed)) FD722_VideoClockCounter;

#define FD722_VideoClockCounter_A (0x88)

__always_inline FD722_VideoClockCounter FD722_VideoClockCounter_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_VideoClockCounter_A);
	return *((FD722_VideoClockCounter *)&r);
}

typedef struct _FD722_SoftID {
	uint32_t id : 32;
} __attribute__((packed)) FD722_SoftID;

#define FD722_SoftID_A (0x8C)

__always_inline FD722_SoftID FD722_SoftID_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_SoftID_A);
	return *((FD722_SoftID *)&r);
}

typedef struct _FD722_ClockFrequency {
	uint32_t frequency : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD722_ClockFrequency;

#define FD722_ClockFrequency_A (0x8D)

__always_inline FD722_ClockFrequency FD722_ClockFrequency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_ClockFrequency_A);
	return *((FD722_ClockFrequency *)&r);
}

__always_inline void FD722_ClockFrequency_W(uint32_t *csr, FD722_ClockFrequency v)
{
	__write_reg(csr, FD722_ClockFrequency_A, *((uint32_t *)&v));
}

typedef struct _FD722_BypassTimeout {
	uint32_t timeout : 32;
} __attribute__((packed)) FD722_BypassTimeout;

#define FD722_BypassTimeout_A(n) (((n == 1) ? 0x91 : 0x90))

__always_inline void FD722_BypassTimeout_W(uint32_t *csr, unsigned int n, FD722_BypassTimeout v)
{
	__write_reg(csr, FD722_BypassTimeout_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_StaticVersion {
	uint32_t revision : 16;
	uint32_t minor : 8;
	uint32_t major : 4;
	uint32_t project : 4;
} __attribute__((packed)) FD722_StaticVersion;

#define FD722_StaticVersion_A (0xFF)

__always_inline FD722_StaticVersion FD722_StaticVersion_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_StaticVersion_A);
	return *((FD722_StaticVersion *)&r);
}

typedef struct _FD722_VDMA {
	uint32_t _unused3000 : 31;
	uint32_t enable : 1;
} __attribute__((packed)) FD722_VDMA;

#define FD722_VDMA_A (0x00)

__always_inline FD722_VDMA FD722_VDMA_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_VDMA_A);
	return *((FD722_VDMA *)&r);
}

__always_inline void FD722_VDMA_W(uint32_t *csr, FD722_VDMA v)
{
	__write_reg(csr, FD722_VDMA_A, *((uint32_t *)&v));
}

typedef struct _FD722_VDMADescriptor {
	uint32_t address : 32;
} __attribute__((packed)) FD722_VDMADescriptor;

#define FD722_VDMADescriptor_A (0x01)

__always_inline FD722_VDMADescriptor FD722_VDMADescriptor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_VDMADescriptor_A);
	return *((FD722_VDMADescriptor *)&r);
}

__always_inline void FD722_VDMADescriptor_W(uint32_t *csr, FD722_VDMADescriptor v)
{
	__write_reg(csr, FD722_VDMADescriptor_A, *((uint32_t *)&v));
}

typedef struct _FD722_IRQEnable {
	uint32_t _unused0300 : 4;
	uint32_t out0IRQ : 1;
	uint32_t _unused0505 : 1;
	uint32_t out1IRQ : 1;
	uint32_t _unused0707 : 1;
	uint32_t sync0IRQ : 1;
	uint32_t _unused0909 : 1;
	uint32_t sync1IRQ : 1;
	uint32_t _unused1511 : 5;
	uint32_t in0IRQ : 1;
	uint32_t _unused1917 : 3;
	uint32_t in1IRQ : 1;
	uint32_t _unused3121 : 11;
} __attribute__((packed)) FD722_IRQEnable;

#define FD722_IRQEnable_A (0x04)

__always_inline FD722_IRQEnable FD722_IRQEnable_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_IRQEnable_A);
	return *((FD722_IRQEnable *)&r);
}

__always_inline void FD722_IRQEnable_W(uint32_t *csr, FD722_IRQEnable v)
{
	__write_reg(csr, FD722_IRQEnable_A, *((uint32_t *)&v));
}

typedef struct _FD722_IRQFlags {
	uint32_t _unused0300 : 4;
	uint32_t out0IRQ : 1;
	uint32_t out0Buffer : 1;
	uint32_t out1IRQ : 1;
	uint32_t out1Buffer : 1;
	uint32_t sync0IRQ : 1;
	uint32_t sync0Field : 1;
	uint32_t sync1IRQ : 1;
	uint32_t sync1Field : 1;
	uint32_t _unused1512 : 4;
	uint32_t in0IRQ : 1;
	uint32_t in0Buffer : 3;
	uint32_t in1IRQ : 1;
	uint32_t in1Buffer : 3;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD722_IRQFlags;

#define FD722_IRQFlags_A (0x05)

__always_inline FD722_IRQFlags FD722_IRQFlags_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_IRQFlags_A);
	return *((FD722_IRQFlags *)&r);
}

__always_inline void FD722_IRQFlags_W(uint32_t *csr, FD722_IRQFlags v)
{
	__write_reg(csr, FD722_IRQFlags_A, *((uint32_t *)&v));
}

typedef struct _FD722_VideoInCS {
	uint32_t cd : 1;
	uint32_t modeLocked : 1;
	uint32_t mode : 2;
	uint32_t reset : 1;
	uint32_t formatLocked : 1;
	uint32_t progressive : 1;
	uint32_t forceMode : 1;
	uint32_t captureRaw : 1;
	uint32_t capture : 1;
	uint32_t _unused1210 : 3;
	uint32_t vbiTop : 1;
	uint32_t vis10b : 1;
	uint32_t vbi10b : 1;
	uint32_t vancCapture : 1;
	uint32_t vbiBottom : 1;
	uint32_t ycSwap : 1;
	uint32_t levelB : 1;
	uint32_t asiPeriod : 2;
	uint32_t _unused3022 : 9;
	uint32_t test : 1;
} __attribute__((packed)) FD722_VideoInCS;

#define FD722_VideoInCS_A(n) (((n == 1) ? 0x0B : 0x06))

__always_inline FD722_VideoInCS FD722_VideoInCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInCS_A(n));
	return *((FD722_VideoInCS *)&r);
}

__always_inline void FD722_VideoInCS_W(uint32_t *csr, unsigned int n, FD722_VideoInCS v)
{
	__write_reg(csr, FD722_VideoInCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoInLine {
	uint32_t totalLines : 12;
	uint32_t _unused1512 : 4;
	uint32_t activeLines : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD722_VideoInLine;

#define FD722_VideoInLine_A(n) (((n == 1) ? 0x0C : 0x07))

__always_inline FD722_VideoInLine FD722_VideoInLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInLine_A(n));
	return *((FD722_VideoInLine *)&r);
}

typedef struct _FD722_VideoInPixel {
	uint32_t totalPixels : 13;
	uint32_t _unused1513 : 3;
	uint32_t activePixels : 13;
	uint32_t _unused3129 : 3;
} __attribute__((packed)) FD722_VideoInPixel;

#define FD722_VideoInPixel_A(n) (((n == 1) ? 0x0D : 0x08))

__always_inline FD722_VideoInPixel FD722_VideoInPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInPixel_A(n));
	return *((FD722_VideoInPixel *)&r);
}

typedef struct _FD722_VideoInIRQTime {
	uint32_t time : 32;
} __attribute__((packed)) FD722_VideoInIRQTime;

#define FD722_VideoInIRQTime_A(n) (((n == 1) ? 0x0E : 0x09))

__always_inline FD722_VideoInIRQTime FD722_VideoInIRQTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInIRQTime_A(n));
	return *((FD722_VideoInIRQTime *)&r);
}

typedef struct _FD722_VideoInIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD722_VideoInIRQRefTime;

#define FD722_VideoInIRQRefTime_A(n) (((n == 1) ? 0x0F : 0x0A))

__always_inline FD722_VideoInIRQRefTime FD722_VideoInIRQRefTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInIRQRefTime_A(n));
	return *((FD722_VideoInIRQRefTime *)&r);
}

__always_inline void FD722_VideoInIRQRefTime_W(uint32_t *csr, unsigned int n,
					       FD722_VideoInIRQRefTime v)
{
	__write_reg(csr, FD722_VideoInIRQRefTime_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoInANCCounter {
	uint32_t counter : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD722_VideoInANCCounter;

#define FD722_VideoInANCCounter_A(n) (((n == 1) ? 0x11 : 0x10))

__always_inline FD722_VideoInANCCounter FD722_VideoInANCCounter_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInANCCounter_A(n));
	return *((FD722_VideoInANCCounter *)&r);
}

typedef struct _FD722_VideoOutCS {
	uint32_t reset : 1;
	uint32_t clone : 1;
	uint32_t mode : 2;
	uint32_t audioCount : 2;
	uint32_t progressive : 1;
	uint32_t freeRunning : 1;
	uint32_t playbackRaw : 1;
	uint32_t playback : 1;
	uint32_t mute : 1;
	uint32_t _unused1111 : 1;
	uint32_t clockM : 1;
	uint32_t vbiTop : 1;
	uint32_t vis10b : 1;
	uint32_t vbi10b : 1;
	uint32_t syncSource : 1;
	uint32_t vbiBottom : 1;
	uint32_t dataPresent : 1;
	uint32_t levelB : 1;
	uint32_t atcEnable : 1;
	uint32_t atcType : 2;
	uint32_t dpllEnable : 1;
	uint32_t dpllSelect : 1;
	uint32_t watchdogEn : 1;
	uint32_t _unused3026 : 5;
	uint32_t test : 1;
} __attribute__((packed)) FD722_VideoOutCS;

#define FD722_VideoOutCS_A(n) (((n == 1) ? 0x1A : 0x12))

__always_inline FD722_VideoOutCS FD722_VideoOutCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutCS_A(n));
	return *((FD722_VideoOutCS *)&r);
}

__always_inline void FD722_VideoOutCS_W(uint32_t *csr, unsigned int n, FD722_VideoOutCS v)
{
	__write_reg(csr, FD722_VideoOutCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoOutLine {
	uint32_t totalLines : 12;
	uint32_t _unused3112 : 20;
} __attribute__((packed)) FD722_VideoOutLine;

#define FD722_VideoOutLine_A(n) (((n == 1) ? 0x1B : 0x13))

__always_inline FD722_VideoOutLine FD722_VideoOutLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutLine_A(n));
	return *((FD722_VideoOutLine *)&r);
}

__always_inline void FD722_VideoOutLine_W(uint32_t *csr, unsigned int n, FD722_VideoOutLine v)
{
	__write_reg(csr, FD722_VideoOutLine_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoOutPixel {
	uint32_t totalPixels : 13;
	uint32_t _unused1513 : 3;
	uint32_t activePixels : 13;
	uint32_t _unused3129 : 3;
} __attribute__((packed)) FD722_VideoOutPixel;

#define FD722_VideoOutPixel_A(n) (((n == 1) ? 0x1C : 0x14))

__always_inline FD722_VideoOutPixel FD722_VideoOutPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutPixel_A(n));
	return *((FD722_VideoOutPixel *)&r);
}

__always_inline void FD722_VideoOutPixel_W(uint32_t *csr, unsigned int n, FD722_VideoOutPixel v)
{
	__write_reg(csr, FD722_VideoOutPixel_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoOutStart {
	uint32_t startOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t startEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD722_VideoOutStart;

#define FD722_VideoOutStart_A(n) (((n == 1) ? 0x1D : 0x15))

__always_inline FD722_VideoOutStart FD722_VideoOutStart_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutStart_A(n));
	return *((FD722_VideoOutStart *)&r);
}

__always_inline void FD722_VideoOutStart_W(uint32_t *csr, unsigned int n, FD722_VideoOutStart v)
{
	__write_reg(csr, FD722_VideoOutStart_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoOutStop {
	uint32_t stopOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t stopEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD722_VideoOutStop;

#define FD722_VideoOutStop_A(n) (((n == 1) ? 0x1E : 0x16))

__always_inline FD722_VideoOutStop FD722_VideoOutStop_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutStop_A(n));
	return *((FD722_VideoOutStop *)&r);
}

__always_inline void FD722_VideoOutStop_W(uint32_t *csr, unsigned int n, FD722_VideoOutStop v)
{
	__write_reg(csr, FD722_VideoOutStop_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoOutField {
	uint32_t switchOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t switchEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD722_VideoOutField;

#define FD722_VideoOutField_A(n) (((n == 1) ? 0x1F : 0x17))

__always_inline FD722_VideoOutField FD722_VideoOutField_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutField_A(n));
	return *((FD722_VideoOutField *)&r);
}

__always_inline void FD722_VideoOutField_W(uint32_t *csr, unsigned int n, FD722_VideoOutField v)
{
	__write_reg(csr, FD722_VideoOutField_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoOutAudio {
	uint32_t address : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD722_VideoOutAudio;

#define FD722_VideoOutAudio_A(n) (((n == 1) ? 0x20 : 0x18))

__always_inline FD722_VideoOutAudio FD722_VideoOutAudio_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutAudio_A(n));
	return *((FD722_VideoOutAudio *)&r);
}

__always_inline void FD722_VideoOutAudio_W(uint32_t *csr, unsigned int n, FD722_VideoOutAudio v)
{
	__write_reg(csr, FD722_VideoOutAudio_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoOutVPID {
	uint32_t vpid : 32;
} __attribute__((packed)) FD722_VideoOutVPID;

#define FD722_VideoOutVPID_A(n) (((n == 1) ? 0x21 : 0x19))

__always_inline FD722_VideoOutVPID FD722_VideoOutVPID_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutVPID_A(n));
	return *((FD722_VideoOutVPID *)&r);
}

__always_inline void FD722_VideoOutVPID_W(uint32_t *csr, unsigned int n, FD722_VideoOutVPID v)
{
	__write_reg(csr, FD722_VideoOutVPID_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_AnalogSync {
	uint32_t hsync : 11;
	uint32_t vsync : 20;
	uint32_t _unused3131 : 1;
} __attribute__((packed)) FD722_AnalogSync;

#define FD722_AnalogSync_A (0x22)

__always_inline FD722_AnalogSync FD722_AnalogSync_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_AnalogSync_A);
	return *((FD722_AnalogSync *)&r);
}

typedef struct _FD722_VideoOutOddDataCount {
	uint32_t count : 24;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD722_VideoOutOddDataCount;

#define FD722_VideoOutOddDataCount_A(n) (((n == 1) ? 0x25 : 0x23))

__always_inline FD722_VideoOutOddDataCount FD722_VideoOutOddDataCount_R(uint32_t *csr,
									unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutOddDataCount_A(n));
	return *((FD722_VideoOutOddDataCount *)&r);
}

__always_inline void FD722_VideoOutOddDataCount_W(uint32_t *csr, unsigned int n,
						  FD722_VideoOutOddDataCount v)
{
	__write_reg(csr, FD722_VideoOutOddDataCount_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoOutEvenDataCount {
	uint32_t count : 24;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD722_VideoOutEvenDataCount;

#define FD722_VideoOutEvenDataCount_A(n) (((n == 1) ? 0x26 : 0x24))

__always_inline FD722_VideoOutEvenDataCount FD722_VideoOutEvenDataCount_R(uint32_t *csr,
									  unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutEvenDataCount_A(n));
	return *((FD722_VideoOutEvenDataCount *)&r);
}

__always_inline void FD722_VideoOutEvenDataCount_W(uint32_t *csr, unsigned int n,
						   FD722_VideoOutEvenDataCount v)
{
	__write_reg(csr, FD722_VideoOutEvenDataCount_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_SyncGenerator {
	uint32_t frameSize : 23;
	uint32_t _unused3023 : 8;
	uint32_t sync : 1;
} __attribute__((packed)) FD722_SyncGenerator;

#define FD722_SyncGenerator_A(n) (((n == 1) ? 0x29 : 0x27))

__always_inline FD722_SyncGenerator FD722_SyncGenerator_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_SyncGenerator_A(n));
	return *((FD722_SyncGenerator *)&r);
}

__always_inline void FD722_SyncGenerator_W(uint32_t *csr, unsigned int n, FD722_SyncGenerator v)
{
	__write_reg(csr, FD722_SyncGenerator_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_SyncGeneratorField {
	uint32_t fieldSize : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD722_SyncGeneratorField;

#define FD722_SyncGeneratorField_A(n) (((n == 1) ? 0x2A : 0x28))

__always_inline FD722_SyncGeneratorField FD722_SyncGeneratorField_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_SyncGeneratorField_A(n));
	return *((FD722_SyncGeneratorField *)&r);
}

__always_inline void FD722_SyncGeneratorField_W(uint32_t *csr, unsigned int n,
						FD722_SyncGeneratorField v)
{
	__write_reg(csr, FD722_SyncGeneratorField_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_Genlock {
	uint32_t mode : 2;
	uint32_t lock : 1;
	uint32_t _unused0803 : 6;
	uint32_t offset : 23;
} __attribute__((packed)) FD722_Genlock;

#define FD722_Genlock_A (0x2B)

__always_inline FD722_Genlock FD722_Genlock_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_Genlock_A);
	return *((FD722_Genlock *)&r);
}

__always_inline void FD722_Genlock_W(uint32_t *csr, FD722_Genlock v)
{
	__write_reg(csr, FD722_Genlock_A, *((uint32_t *)&v));
}

typedef struct _FD722_SyncGeneratorTime {
	uint32_t time : 32;
} __attribute__((packed)) FD722_SyncGeneratorTime;

#define FD722_SyncGeneratorTime_A(n) (((n == 1) ? 0x2D : 0x2C))

__always_inline FD722_SyncGeneratorTime FD722_SyncGeneratorTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_SyncGeneratorTime_A(n));
	return *((FD722_SyncGeneratorTime *)&r);
}

__always_inline void FD722_SyncGeneratorTime_W(uint32_t *csr, unsigned int n,
					       FD722_SyncGeneratorTime v)
{
	__write_reg(csr, FD722_SyncGeneratorTime_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoInFIFOStatus {
	uint32_t videoMax : 15;
	uint32_t videoError : 1;
	uint32_t audioMax : 15;
	uint32_t audioError : 1;
} __attribute__((packed)) FD722_VideoInFIFOStatus;

#define FD722_VideoInFIFOStatus_A(n) (((n == 1) ? 0x2F : 0x2E))

__always_inline FD722_VideoInFIFOStatus FD722_VideoInFIFOStatus_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInFIFOStatus_A(n));
	return *((FD722_VideoInFIFOStatus *)&r);
}

typedef struct _FD722_VideoOutFIFOStatus {
	uint32_t videoMin : 15;
	uint32_t videoError : 1;
	uint32_t audioMin : 15;
	uint32_t audioError : 1;
} __attribute__((packed)) FD722_VideoOutFIFOStatus;

#define FD722_VideoOutFIFOStatus_A(n) (((n == 1) ? 0x31 : 0x30))

__always_inline FD722_VideoOutFIFOStatus FD722_VideoOutFIFOStatus_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutFIFOStatus_A(n));
	return *((FD722_VideoOutFIFOStatus *)&r);
}

typedef struct _FD722_PCIEPerfomance {
	uint32_t txCounter : 16;
	uint32_t rxCounter : 16;
} __attribute__((packed)) FD722_PCIEPerfomance;

#define FD722_PCIEPerfomance_A (0x32)

__always_inline FD722_PCIEPerfomance FD722_PCIEPerfomance_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_PCIEPerfomance_A);
	return *((FD722_PCIEPerfomance *)&r);
}

typedef struct _FD722_VideoInFrameSize {
	uint32_t size : 21;
	uint32_t _unused3121 : 11;
} __attribute__((packed)) FD722_VideoInFrameSize;

#define FD722_VideoInFrameSize_A(n) (((n == 1) ? 0x34 : 0x33))

__always_inline FD722_VideoInFrameSize FD722_VideoInFrameSize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInFrameSize_A(n));
	return *((FD722_VideoInFrameSize *)&r);
}

typedef struct _FD722_SyncGeneratorPhase {
	uint32_t phase : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD722_SyncGeneratorPhase;

#define FD722_SyncGeneratorPhase_A(n) (((n == 1) ? 0x36 : 0x35))

__always_inline void FD722_SyncGeneratorPhase_W(uint32_t *csr, unsigned int n,
						FD722_SyncGeneratorPhase v)
{
	__write_reg(csr, FD722_SyncGeneratorPhase_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoInVPID {
	uint32_t vpid : 32;
} __attribute__((packed)) FD722_VideoInVPID;

#define FD722_VideoInVPID_A(n) (((n == 1) ? 0x38 : 0x37))

__always_inline FD722_VideoInVPID FD722_VideoInVPID_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInVPID_A(n));
	return *((FD722_VideoInVPID *)&r);
}

typedef struct _FD722_PCIERxMaxLatency {
	uint32_t maxLatency : 16;
	uint32_t numRequests : 16;
} __attribute__((packed)) FD722_PCIERxMaxLatency;

#define FD722_PCIERxMaxLatency_A (0x39)

__always_inline FD722_PCIERxMaxLatency FD722_PCIERxMaxLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_PCIERxMaxLatency_A);
	return *((FD722_PCIERxMaxLatency *)&r);
}

typedef struct _FD722_PCIERxTotalLatency {
	uint32_t latency : 32;
} __attribute__((packed)) FD722_PCIERxTotalLatency;

#define FD722_PCIERxTotalLatency_A (0x3A)

__always_inline FD722_PCIERxTotalLatency FD722_PCIERxTotalLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_PCIERxTotalLatency_A);
	return *((FD722_PCIERxTotalLatency *)&r);
}

#define FD722_VideoOutATC_A(n) (((n == 1) ? 0x3C : 0x3B))

typedef struct _FD722_AnalogVSyncTime {
	uint32_t time : 32;
} __attribute__((packed)) FD722_AnalogVSyncTime;

#define FD722_AnalogVSyncTime_A (0x3F)

__always_inline FD722_AnalogVSyncTime FD722_AnalogVSyncTime_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_AnalogVSyncTime_A);
	return *((FD722_AnalogVSyncTime *)&r);
}

typedef struct _FD722_ClockDeviation {
	uint32_t deviation : 24;
	uint32_t _unused3024 : 7;
	uint32_t direction : 1;
} __attribute__((packed)) FD722_ClockDeviation;

#define FD722_ClockDeviation_A (0x8B)

__always_inline FD722_ClockDeviation FD722_ClockDeviation_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_ClockDeviation_A);
	return *((FD722_ClockDeviation *)&r);
}

__always_inline void FD722_ClockDeviation_W(uint32_t *csr, FD722_ClockDeviation v)
{
	__write_reg(csr, FD722_ClockDeviation_A, *((uint32_t *)&v));
}
typedef struct _FD722_VideoInVBISize {
	uint32_t size : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD722_VideoInVBISize;

#define FD722_VideoInVBISize_A(n) (((n == 1) ? 0x4A : 0x48))

__always_inline FD722_VideoInVBISize FD722_VideoInVBISize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInVBISize_A(n));
	return *((FD722_VideoInVBISize *)&r);
}

typedef struct _FD722_VideoInVisibleSize {
	uint32_t size : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD722_VideoInVisibleSize;

#define FD722_VideoInVisibleSize_A(n) (((n == 1) ? 0x4B : 0x49))

__always_inline FD722_VideoInVisibleSize FD722_VideoInVisibleSize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoInVisibleSize_A(n));
	return *((FD722_VideoInVisibleSize *)&r);
}

typedef struct _FD722_PPSLTCControl {
	uint32_t mode : 2;
	uint32_t _unused1502 : 14;
	uint32_t valid : 1;
	uint32_t _unused3117 : 15;
} __attribute__((packed)) FD722_PPSLTCControl;

#define FD722_PPSLTCControl_A (0x4C)

__always_inline FD722_PPSLTCControl FD722_PPSLTCControl_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_PPSLTCControl_A);
	return *((FD722_PPSLTCControl *)&r);
}

__always_inline void FD722_PPSLTCControl_W(uint32_t *csr, FD722_PPSLTCControl v)
{
	__write_reg(csr, FD722_PPSLTCControl_A, *((uint32_t *)&v));
}

typedef struct _FD722_PPSLTCTimestamp {
	uint32_t time : 32;
} __attribute__((packed)) FD722_PPSLTCTimestamp;

#define FD722_PPSLTCTimestamp_A (0x4D)

__always_inline FD722_PPSLTCTimestamp FD722_PPSLTCTimestamp_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_PPSLTCTimestamp_A);
	return *((FD722_PPSLTCTimestamp *)&r);
}

typedef struct _FD722_PPSLTCDataLow {
	uint32_t data : 32;
} __attribute__((packed)) FD722_PPSLTCDataLow;

#define FD722_PPSLTCDataLow_A (0x4E)

__always_inline FD722_PPSLTCDataLow FD722_PPSLTCDataLow_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_PPSLTCDataLow_A);
	return *((FD722_PPSLTCDataLow *)&r);
}

typedef struct _FD722_PPSLTCDataHigh {
	uint32_t data : 32;
} __attribute__((packed)) FD722_PPSLTCDataHigh;

#define FD722_PPSLTCDataHigh_A (0x4F)

__always_inline FD722_PPSLTCDataHigh FD722_PPSLTCDataHigh_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_PPSLTCDataHigh_A);
	return *((FD722_PPSLTCDataHigh *)&r);
}

typedef struct _FD722_VideoOutIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD722_VideoOutIRQRefTime;

#define FD722_VideoOutIRQRefTime_A(n) (((n == 1) ? 0x51 : 0x50))

__always_inline FD722_VideoOutIRQRefTime FD722_VideoOutIRQRefTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutIRQRefTime_A(n));
	return *((FD722_VideoOutIRQRefTime *)&r);
}

typedef struct _FD722_VideoOutPhaseShift {
	uint32_t phase : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD722_VideoOutPhaseShift;

#define FD722_VideoOutPhaseShift_A(n) (((n == 1) ? 0x53 : 0x52))

__always_inline FD722_VideoOutPhaseShift FD722_VideoOutPhaseShift_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD722_VideoOutPhaseShift_A(n));
	return *((FD722_VideoOutPhaseShift *)&r);
}

__always_inline void FD722_VideoOutPhaseShift_W(uint32_t *csr, unsigned int n,
						FD722_VideoOutPhaseShift v)
{
	__write_reg(csr, FD722_VideoOutPhaseShift_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_VideoOutWatchdog {
	uint32_t timeout : 16;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD722_VideoOutWatchdog;

#define FD722_VideoOutWatchdog_A(n) (((n == 1) ? 0x55 : 0x54))

__always_inline void FD722_VideoOutWatchdog_W(uint32_t *csr, unsigned int n,
					      FD722_VideoOutWatchdog v)
{
	__write_reg(csr, FD722_VideoOutWatchdog_A(n), *((uint32_t *)&v));
}

typedef struct _FD722_M2VideoIODir {
	uint32_t dir : 1;
	uint32_t _unused3101 : 31;
} __attribute__((packed)) FD722_M2VideoIODir;

#define FD722_M2VideoIODir_A (0x7E)

__always_inline FD722_M2VideoIODir FD722_M2VideoIODir_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_M2VideoIODir_A);
	return *((FD722_M2VideoIODir *)&r);
}

__always_inline void FD722_M2VideoIODir_W(uint32_t *csr, FD722_M2VideoIODir v)
{
	__write_reg(csr, FD722_M2VideoIODir_A, *((uint32_t *)&v));
}

typedef struct _FD722_DynamicVersion {
	uint32_t revision : 16;
	uint32_t minor : 8;
	uint32_t major : 4;
	uint32_t project : 4;
} __attribute__((packed)) FD722_DynamicVersion;

#define FD722_DynamicVersion_A (0x7F)

__always_inline FD722_DynamicVersion FD722_DynamicVersion_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD722_DynamicVersion_A);
	return *((FD722_DynamicVersion *)&r);
}

#endif
